#include <math.h>
#include <string>
#include "../projection.hpp"
#include "../Texture.hpp"
#include "../Node.hpp"
#include "rawimage.hpp"

using namespace std;

int main( int argc, char **argv)
{
   int base_lod = -1;
   char face = '0';
   unsigned int input_width = 0;
   unsigned int input_height = 0;
   unsigned int output_width = 0;
   unsigned int output_height = 0;
   string input_raster_filename;
   string output_prefix;
   
   // Process parameters

   for(int a=1;a<argc;a++) 
     {
	if ( strcmp( argv[a], "--help") == 0 ) 
	  {
	     fprintf( stdout, "usage: %s [options] <input_file> <output_prefix>\n\n", argv[0]);
	     fprintf( stdout, "options are:\n");
	     fprintf( stdout, "  --help              Display this help notice\n");
	     fprintf( stdout, "  --base-lod <lod>    Build quadtree from base lod\n");
	     fprintf( stdout, "  --face <id>         Select face to build: N|S|F|B|E|W\n");
	     fprintf( stderr, "  --input-dim         <width>x<height>\n");
	     fprintf( stdout, "  --output-dim        <width>x<height>\n");
	     fprintf( stderr, "  --inpuut-raster <file>\n");
	     fprintf( stdout, "  --output-prefix <prefix>\n");
	     
	     exit( EXIT_FAILURE);
	  }
	else if ( strcmp( argv[a], "--base-lod") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  base_lod = atol( argv[++a]);
	       }
	     else
	       {
		  fprintf( stderr, "--base-lod must be followed by an integer\n");
		  exit( EXIT_FAILURE);
	       }
	  }
	else if ( strcmp( argv[a], "--face") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  face = argv[++a][0];
	       }
	     else
	       {
		  fprintf( stderr, "--face must take a face id\n");
		  exit( EXIT_FAILURE);
	       }
	  }
		else if ( strcmp( argv[a], "--input-dim") == 0 )
	  {
	     if ( a+1 < argc )
	       {		  
		
		  if ( sscanf( argv[++a], "%dx%d", &input_width, &input_height) != 2)
		    {
		       fprintf( stderr, "%s is a malformed dimension ( should be <width>x<height> )\n", argv[a]);
		       exit( EXIT_FAILURE);	       
		    }
	       }
	     else
	       {
		  fprintf( stderr, "--input-dim must take a dimension of the form <width>x<height>\n");
		  exit( EXIT_FAILURE);
	       }
	  }
	else if ( strcmp( argv[a], "--output-dim") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  
		  if ( sscanf( argv[++a], "%dx%d", &output_width, &output_height) != 2)
		    {
		       fprintf( stderr, "%s is a malformed dimension ( should be <width>x<height> )\n", argv[a]);
		       exit( EXIT_FAILURE);	       
		    }
	       }
	     else
	       {
		  fprintf( stderr, "--output-dim must take a dimension of the form <width>x<height>\n");
		  exit( EXIT_FAILURE);
	       }
	  }
	else if ( strcmp( argv[a], "--input-raster") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  input_raster_filename = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--input-raster must take a filename\n");
		  exit( EXIT_FAILURE);
	       }
	  }	
	else if ( strcmp( argv[a], "--output-prefix") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  output_prefix = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--output-prefix must take a prefix path\n");
		  exit( EXIT_FAILURE);
	       }
	  }	

     }
   
   if ( face == '0' )
     {
	fprintf( stderr, "%s: please specify a face ( --face <id> )\n", argv[0]);
	exit( EXIT_FAILURE);
     }
   
   if ( base_lod == -1 )
     {
     	fprintf( stderr, "%s: please specify a base lod ( --base-lod <lod> )\n", argv[0]);
	exit( EXIT_FAILURE);
     }
   
   Cube::Face current_face;
   
   try
     {
	current_face = Cube::GetFace( face);
     }
   catch(...)
     {
	fprintf( stderr, "%s: face %c is not a valid face identifier\n", argv[0]);
	exit( EXIT_FAILURE);
     }
   
   // Load original raster
   
   RawImage input_raster;
   input_raster.OpenFile( input_raster_filename.c_str(), input_width, input_height);
   input_raster.Map();

   unsigned int base_width = 1 << base_lod;
   
   fprintf( stderr, "%s: Requested to build a quadtree made of %d lod\n", argv[0], base_lod+1);
   
   // Create destination Texture
   
   Texture node_texture;
   node_texture.CreateFromScratch( output_width, output_height, GL_RGB);
   
   // TODO: x_step and y_step must be equal to 1 
   // without it, we need to interpolate ( bilinear ? )
   float x_step = ((float)input_width)  / ((float)output_width)  / ((float)base_width); 
   float y_step = ((float)input_height) / ((float)output_height) / ((float)base_width);
   float gx_step = x_step * (float) output_width;
   float gy_step = y_step * (float) output_height;

   if ( x_step != 1.0f || y_step != 1.0f)
     {
	fprintf( stderr, "%s: [WARNING] requested dimension of the base is not the same as the input raster dimension\n", argv[0]); 
	fprintf( stderr, "Please, check your parameters\n");
	fprintf( stderr, "( step_x = %f    step_y = %f )\n", x_step, y_step);
     }

   fprintf( stderr, "Building base level %d (%dx%d nodes)\n", base_lod, base_width, base_width);

   // Sweep all nodes from quadtree's base
   
   for ( unsigned int gy = 0 ; gy < base_width ; gy++ )
     for ( unsigned int gx=0 ; gx < base_width ; gx++ )
       {
	  
	  float gx_offset = gx_step * (float) gx;
	  float gy_offset = gy_step * (float) gy; 
	
	  // Fill destination texture
	  
	  for ( unsigned int y = 0 ; y < output_height ; y++ )
	    for ( unsigned int x = 0 ; x < output_width ; x++ )
	      {
		 
		 unsigned int ox = (unsigned int) ( gx_offset + (float) x * x_step );
		 unsigned int oy = input_height - (unsigned int) ( gy_offset + (float) y * y_step );
		 
		 node_texture.SetPixelComp( x, y, Texture::B, input_raster.GetPixelComp( ox, oy, RawImage::R) );
		 node_texture.SetPixelComp( x, y, Texture::G, input_raster.GetPixelComp( ox, oy, RawImage::G) );
		 node_texture.SetPixelComp( x, y, Texture::R, input_raster.GetPixelComp( ox, oy, RawImage::B) );
		 
	      }
	  
	  // Save node texture to disk
	  
	  Node node = Node( base_lod, gx, gy, current_face);
	  
	  char filename[128];
	  
	  sprintf( filename, "%s/%c/%04d.bmp", output_prefix.c_str(), Cube::GetFace( current_face), node.GetNumber());
	  node_texture.Save( filename);
	  
	  fprintf( stdout, "\rnode %4d saved  ( %4d / %4d )   ", node.GetNumber(), gy*base_width+gx, base_width*base_width+1); 
	  fflush( stdout);
       }
   
   // then build up quadtree
   
   for ( int level = base_lod-1 ; level >= 0 ; level--)
     {
	unsigned int level_width = powf( 2, level);
	
	fprintf( stderr, "\nBuilding up level %d (%dx%d nodes)\n", level, level_width, level_width); 
	
	for( unsigned int y = 0 ; y < level_width ; y++)
	  for( unsigned int x = 0 ; x < level_width ; x++)
	    {
	       Node node = Node( level, x, y, current_face);
	       
	       // Load the 4 relevant textures from level-1  
	       
	       char tmp[1024];
	       
	       Texture t1;
	       sprintf( tmp, "%s/%c/%04d.bmp", output_prefix.c_str(), face, node.GetChild(0).GetNumber());
	       t1.LoadImage( tmp);
	       
	       Texture t2;
	       sprintf( tmp, "%s/%c/%04d.bmp", output_prefix.c_str(), face, node.GetChild(1).GetNumber());
	       t2.LoadImage( tmp);
	
	       Texture t3;
	       sprintf( tmp, "%s/%c/%04d.bmp", output_prefix.c_str(), face, node.GetChild(2).GetNumber());
	       t3.LoadImage( tmp);
	       
	       Texture t4;
	       sprintf( tmp, "%s/%c/%04d.bmp", output_prefix.c_str(), face, node.GetChild(3).GetNumber());
	       t4.LoadImage( tmp);
		       
	       // Create intermediate Texture
	       Texture intermediate;
	       intermediate.CreateFromScratch( output_width*2, output_height*2, GL_RGB);
	       
	       for (unsigned int py=0 ; py < output_height ; py++)
		 for (unsigned int px=0 ; px < output_width ; px++)
		   {
		      intermediate.SetPixelComp( px, output_height*2-1-py, Texture::R, t3.GetPixelComp( px, 511-py, Texture::R));
		      intermediate.SetPixelComp( px, output_height*2-1-py, Texture::G, t3.GetPixelComp( px, 511-py, Texture::G));
		      intermediate.SetPixelComp( px, output_height*2-1-py, Texture::B, t3.GetPixelComp( px, 511-py, Texture::B));

		      intermediate.SetPixelComp( px+output_width, output_height*2-1-py, Texture::R, t4.GetPixelComp( px, 511-py, Texture::R));
		      intermediate.SetPixelComp( px+output_width, output_height*2-1-py, Texture::G, t4.GetPixelComp( px, 511-py, Texture::G));
		      intermediate.SetPixelComp( px+output_width, output_height*2-1-py, Texture::B, t4.GetPixelComp( px, 511-py, Texture::B));

		      intermediate.SetPixelComp( px, output_height*2-1-(py+output_height), Texture::R, t1.GetPixelComp( px, 511-py, Texture::R));
		      intermediate.SetPixelComp( px, output_height*2-1-(py+output_height), Texture::G, t1.GetPixelComp( px, 511-py, Texture::G));
		      intermediate.SetPixelComp( px, output_height*2-1-(py+output_height), Texture::B, t1.GetPixelComp( px, 511-py, Texture::B));

		      intermediate.SetPixelComp( px+output_width, output_height*2-1-(py+output_height), Texture::R, t2.GetPixelComp( px, 511-py, Texture::R));
		      intermediate.SetPixelComp( px+output_width, output_height*2-1-(py+output_height), Texture::G, t2.GetPixelComp( px, 511-py, Texture::G));
		      intermediate.SetPixelComp( px+output_width, output_height*2-1-(py+output_height), Texture::B, t2.GetPixelComp( px, 511-py, Texture::B));
		   }
	       
	       // Create final texture
	       Texture final;
	       final.CreateFromScratch( output_width, output_height, GL_RGB);
	       
	       //gluScaleImage ( GL_RGB, 512, 512, GL_UNSIGNED_BYTE, intermediate.GetPixelsPtr(), 256, 256, GL_UNSIGNED_BYTE, final.GetPixelsPtr());
	       
	       //gluScaleImage (GLenum format, GLsizei wIn, GLsizei hIn, GLenum typeIn, const void *dataIn, GLsizei wOut, GLsizei hOut, GLenum typeOut, GLvoid* dataOut);

	       //intermediate.Resample( output_width, output_height);

	       for (unsigned int py=0 ; py < output_height ; py++)
		 for (unsigned int px=0 ; px < output_width ; px++)
		   {
		      final.SetPixelComp( px, py, Texture::R, 
					  ((int)intermediate.GetPixelComp( px*2,   py*2,   Texture::R)+
					  (int)intermediate.GetPixelComp( px*2+1, py*2,   Texture::R)+
					  (int)intermediate.GetPixelComp( px*2+1, py*2+1, Texture::R)+
					  (int)intermediate.GetPixelComp( px*2,   py*2+1, Texture::R))/4); 

		      final.SetPixelComp( px, py, Texture::G, 
					  ((int)intermediate.GetPixelComp( px*2,   py*2,   Texture::G)+
					  (int)intermediate.GetPixelComp( px*2+1, py*2,   Texture::G)+
					  (int)intermediate.GetPixelComp( px*2+1, py*2+1, Texture::G)+
					  (int)intermediate.GetPixelComp( px*2,   py*2+1, Texture::G))/4);

		      final.SetPixelComp( px, py, Texture::B, 
					  ((int)intermediate.GetPixelComp( px*2,   py*2,   Texture::B)+
					  (int)intermediate.GetPixelComp( px*2+1, py*2,   Texture::B)+
					  (int)intermediate.GetPixelComp( px*2+1, py*2+1, Texture::B)+
					  (int)intermediate.GetPixelComp( px*2,   py*2+1, Texture::B))/4);

		      
		   }
	       
	       // Save final Texture
	       char filename[1024];
	       sprintf( filename, "%s/%c/%04d.bmp", output_prefix.c_str(), face, node.GetNumber());
	       final.Save( filename);
	       
	       fprintf( stdout, "\rnode %4d ( %4d / %4d )    ", node.GetNumber(), y*level_width+x+1, level_width*level_width); 
	       fflush( stdout);
	    }   
     }   
}
