#include <math.h>
#include <vector>
#include "../projection.hpp"
#include "../Node.hpp"
#include "rawimage.hpp"

using namespace std;

int main( int argc, char **argv)
{
   unsigned int input_width =  0;
   unsigned int input_height = 0;
   unsigned int output_width =  0;
   unsigned int output_height = 0;
   Cube::Face face_to_project;
   bool project_all_faces = true;
   string input_raster_filename;
   string input_second_raster_filename;
   bool input_raster_have_2_parts = false;
   string output_prefix;
   
   // Process parameters

   for(int a=1;a<argc;a++) 
     {
	if ( strcmp( argv[a], "--help") == 0 ) 
	  {
	     fprintf( stdout, "usage: %s [options]\n\n", argv[0]);
	     fprintf( stdout, "options are:\n");
	     fprintf( stdout, "  --help                Display this help notice\n");
	     fprintf( stdout, "  --face id             Select face to build: N|S|F|B|E|W (default all planes)\n"); 
	     fprintf( stdout, "  --input-dim <wxh>     Set input raster resolution\n");
	     fprintf( stdout, "                        example: --input-dim 1024x1024\n");
	     fprintf( stdout, "  --output-dim <wxh>    Set output raster resolution\n");
	     fprintf( stdout, "                        example: --input-dim 1024x1024\n");
	     fprintf( stdout, "  --pixel-format <pf>   Set pixel format\n"); 
	     fprintf( stdout, "                        RGB  = 3x8 bits Red,Green,Blue\n");
	     fprintf( stderr, "                        G    = 1x8 bits Grayscale\n");
	     fprintf( stderr, "  --input-raster        must be followed by the filename\n");
	     fprintf( stderr, "  --input-second-raster When input rasters are  west / east separated\n");
	     fprintf( stderr, "                        must be followed by the filename of the second raster\n");
	     fprintf( stderr, "  --output-prefix <prefix>\n");
	     
	     exit( EXIT_FAILURE);
	  }
	else if ( strcmp( argv[a], "--face") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  try
		    {
		       face_to_project = Cube::GetFace( argv[++a][0]);
		    }
		  catch(...)
		    {
		       fprintf( stderr, "%s: %c is an invalid face identifier\n", argv[a-1][0]);
		       exit( EXIT_FAILURE);
		    }
		  project_all_faces = false;
	       }
	     else
	       {
		  fprintf( stderr, "--face must take a face id\n");
		  exit( EXIT_FAILURE);
	       }
	  }
	else if ( strcmp( argv[a], "--input-dim") == 0 )
	  {
	     if ( a+1 < argc )
	       {		  
		
		  if ( sscanf( argv[++a], "%dx%d", &input_width, &input_height) != 2)
		    {
		       fprintf( stderr, "%s is a malformed dimension ( should be <width>x<height> )\n", argv[a]);
		       exit( EXIT_FAILURE);	       
		    }
	       }
	     else
	       {
		  fprintf( stderr, "--input-dim must take a dimension of the form <width>x<height>\n");
		  exit( EXIT_FAILURE);
	       }
	  }
	else if ( strcmp( argv[a], "--output-dim") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  
		  if ( sscanf( argv[++a], "%dx%d", &output_width, &output_height) != 2)
		    {
		       fprintf( stderr, "%s is a malformed dimension ( should be <width>x<height> )\n", argv[a]);
		       exit( EXIT_FAILURE);	       
		    }
	       }
	     else
	       {
		  fprintf( stderr, "--output-dim must take a dimension of the form <width>x<height>\n");
		  exit( EXIT_FAILURE);
	       }
	  }
	else if ( strcmp( argv[a], "--input-raster") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  input_raster_filename = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--input-raster must take a filename\n");
		  exit( EXIT_FAILURE);
	       }	     
	  }	
	else if ( strcmp( argv[a], "--input-second-raster") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  input_second_raster_filename = argv[++a];
		  input_raster_have_2_parts = true;
	       }
	     else
	       {
		  fprintf( stderr, "--input-second-raster must take a filename\n");
		  exit( EXIT_FAILURE);
	       }	     
	  }
	else if ( strcmp( argv[a], "--output-prefix") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  output_prefix = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--output-prefix must take a path prefix\n");
		  exit( EXIT_FAILURE);
	       }	     	     
	  }    
     }
   
   // check for required parameters
   
   if ( input_width == 0 || input_height == 0 )
     {
	fprintf( stderr, "%s: please set input dimension ( --input-dim <width>x<height> )\n", argv[0]);
	exit( EXIT_FAILURE);
     }
   
   if ( output_width == 0 || output_height == 0 )
     {
	fprintf( stderr, "%s: please set output dimension ( --output-dim <width>x<height> )\n", argv[0]);
	exit( EXIT_FAILURE);
     }

   if ( input_raster_filename == string("") )
     {
     	fprintf( stderr, "%s: please set input raster file ( --input-raster <raster.raw> )\n", argv[0]);
	exit( EXIT_FAILURE);
     }
   
   if ( output_prefix == string("") )
     {
     	fprintf( stderr, "%s: please set output prefix ( --output-prefix <prefix> )\n", argv[0]);
	exit( EXIT_FAILURE);
     }  
   
   // Open the input raster(s)
   
   RawImage ri_west;
   RawImage ri_east;

   ri_west.OpenFile( input_raster_filename.c_str(), input_raster_have_2_parts?input_width/2:input_width, input_height);
   ri_west.Map();
   
   if ( input_raster_have_2_parts )
     {
	ri_east.OpenFile( input_second_raster_filename.c_str(), input_width/2, input_height); 
	//ri_east.Map();
     }
   
   vector< Cube::Face> faces_to_project;
   
   if ( project_all_faces)
     {
	faces_to_project.push_back( Cube::Front);
	faces_to_project.push_back( Cube::West);
	faces_to_project.push_back( Cube::Back);
	faces_to_project.push_back( Cube::East);
	faces_to_project.push_back( Cube::North);
	faces_to_project.push_back( Cube::South);
     }
   else
     {
	faces_to_project.push_back( face_to_project);
     }
   
   for( unsigned int f = 0 ; f < faces_to_project.size() ; f++ )
     {
	Cube::Face current_face = faces_to_project[f];
       
	string output_filename = output_prefix + string("_") + Cube::GetFace( current_face) + string( ".raw");
	FILE *output_raw_raster = fopen( output_filename.c_str(), "w");
	
	if ( output_raw_raster == NULL )
	  {
	     fprintf( stderr, "%s: can't open %s for writting", argv[0], output_filename.c_str());
	     continue;
	  }

	fprintf( stdout, "%s: saving %s\n", argv[0], output_filename.c_str());

	float y_step = (2.0f*CUBE_RAY) / (float)output_height;
	float x_step = (2.0f*CUBE_RAY) / (float)output_width;
	
	for( unsigned int y = 0 ; y < output_height ; y++ )
	  {
	     fprintf( stdout, "\r %3d%%", 100 * y / output_height);
	     fflush( stdout);
				
	     for( unsigned int x = 0 ; x < output_width ; x++ )
	       {
		  
		  Cube::Coord coord( -CUBE_RAY+((float)x)*x_step,
				     -CUBE_RAY+((float)y)*y_step,
				     current_face );
		  
		  float lat_deg, lon_deg;
		  
		  Coord3DToPolar( coord.GetCoord3D( 1.0f), &lat_deg, &lon_deg);
		  
		  unsigned int ox = (unsigned int) (( lon_deg + M_PI ) / ( 2.0f * M_PI ) * (float)input_width);
		  unsigned int oy = input_height - (unsigned int) (( lat_deg + M_PI/2.0f ) / ( M_PI) * (float)input_height);
		  
		  unsigned char pixel[3];
		  
		  if ( input_raster_have_2_parts)
		    {
		       if ( ox < input_width/2 )
			 {		      
			    pixel[0] = ri_west.GetPixelComp( ox, oy, RawImage::R);
			    pixel[1] = ri_west.GetPixelComp( ox, oy, RawImage::G);
			    pixel[2] = ri_west.GetPixelComp( ox, oy, RawImage::B);
			 }
		       else
			 {
			    pixel[0] = ri_east.GetPixelComp( ox-input_width/2, oy, RawImage::R);
			    pixel[1] = ri_east.GetPixelComp( ox-input_width/2, oy, RawImage::G);
			    pixel[2] = ri_east.GetPixelComp( ox-input_width/2, oy, RawImage::B);   
			 }
		    }
		  else
		    {
		       pixel[0] = ri_west.GetPixelComp( ox, oy, RawImage::R);
		       pixel[1] = ri_west.GetPixelComp( ox, oy, RawImage::G);
		       pixel[2] = ri_west.GetPixelComp( ox, oy, RawImage::B);
		    }
		  
		  fwrite( pixel, 1, 3, output_raw_raster);
	       }
	  }
	
	fprintf( stdout, "\n");
	fclose( output_raw_raster);	
     }
}
