#include <math.h>
#include "../projection.hpp"
#include "../Texture.hpp"
#include "../Node.hpp"
#include "rawimage.hpp"

#define RESOLUTION_W 512
#define RESOLUTION_H 512

#define ORTHO_RESOLUTION_W 21600*2 ///2
#define ORTHO_RESOLUTION_H 21600 ///2
  
#define DEFAULT_BASE_LOD 6

#define DEFAULT_PLANE 'N'

int main( int argc, char **argv)
{

   int base_lod = DEFAULT_BASE_LOD;
   char PLANE = DEFAULT_PLANE;
   
   /* Process parameters */

   for(int a=1;a<argc;a++) 
     {
	if ( strcmp( argv[a], "--help") == 0 ) 
	  {
	     fprintf( stdout, "usage: %s [options]\n\n", argv[0]) ;
	     fprintf( stdout, "options are:\n") ;
	     fprintf( stdout, "  --help           Display this help notice\n") ;
	     fprintf( stdout, "  --base-lod lod   Build level lod (default %d)\n", DEFAULT_PLANE) ;
	     fprintf( stdout, "  --plane id       Select plane to build: N|S|F|B|E|W\n");
	     return EXIT_FAILURE;
	  }
	else if ( strcmp( argv[a], "--base-lod") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  base_lod = atol( argv[++a]);
	       }
	     else
	       {
		  fprintf( stderr, "--base-lod must be followed by an integer\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--plane") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  PLANE = argv[++a][0];
	       }
	     else
	       {
		  fprintf( stderr, "--plane must take a plane id\n");
		  return EXIT_FAILURE;
	       }
	  }
     }
   
   int base_width = 1 << base_lod;
   
   fprintf( stderr, "Will build the base (%dx%d) of a quadtree made of %d lod\n", base_width, base_width, base_lod);
   
   fprintf( stderr, "Building plane %c\n", PLANE);
   
   
   RawImage ri_west;
   ri_west.OpenFile( "/mnt/win_d/BLUE_MARBLE_DATA/BM_west.raw", ORTHO_RESOLUTION_W/2, ORTHO_RESOLUTION_H);
   ri_west.Map();
   
   RawImage ri_east;
   ri_east.OpenFile( "/mnt/win_d/BLUE_MARBLE_DATA/BM_east.raw", ORTHO_RESOLUTION_W/2, ORTHO_RESOLUTION_H); 
//   ri_east.Map();
   
   // Create destination Texture
   Texture cobe;
   cobe.CreateFromScratch( RESOLUTION_W, RESOLUTION_H, GL_RGB);
   
   float x_step = 2.0f * CUBE_RAY / (float)((RESOLUTION_W) * base_width); // -1 to keep same borders in adjacents chunks
   float y_step = 2.0f * CUBE_RAY / (float)((RESOLUTION_H) * base_width); // idem
   
   for(unsigned int gy=0;gy<base_width;gy++)
     for(unsigned int gx=0;gx<base_width;gx++)
       {
	  
	  // Fill destination texture
	  for(unsigned int y=0;y<RESOLUTION_H;y++)
	    for( unsigned int x=0;x<RESOLUTION_W;x++)
	      {
		 // front
		 float lat, lon;
		 
		 //Coord3DToPolar( Coord3D( -CUBE_RAY+((float)x)*x_step, -CUBE_RAY+((float)y)*y_step, CUBE_RAY), &lat, &lon);	  
		 
		 Coord3D cube_coord;
		 
		 switch( PLANE)
		   {
		    case 'N':
		      // North
		      cube_coord.c[0] = -CUBE_RAY+((float)(gx*RESOLUTION_W+x))*x_step;
		      cube_coord.c[1] =  CUBE_RAY; 
		      cube_coord.c[2] = -CUBE_RAY+((float)(gy*RESOLUTION_H+y))*y_step;
		      break;

		    case 'S':
		      // South
		      cube_coord.c[0] = -CUBE_RAY+((float)(gx*RESOLUTION_W+x))*x_step;
		      cube_coord.c[1] = -CUBE_RAY; 
		      cube_coord.c[2] = -CUBE_RAY+((float)(gy*RESOLUTION_H+y))*y_step;
		      break;

		    case 'F':
		      // Front 
		      cube_coord.c[0] = -CUBE_RAY+((float)(gx*RESOLUTION_W+x))*x_step;
		      cube_coord.c[1] =  CUBE_RAY-((float)(gy*RESOLUTION_H+y))*y_step; 
		      cube_coord.c[2] =  CUBE_RAY;
		      break;

		    case 'B':
		      // Back 
		      cube_coord.c[0] = -CUBE_RAY+((float)(gx*RESOLUTION_W+x))*x_step;
		      cube_coord.c[1] =  CUBE_RAY-((float)(gy*RESOLUTION_H+y))*y_step; 
		      cube_coord.c[2] = -CUBE_RAY;
		      break;

		    case 'W':
		      // West
		      cube_coord.c[0] = -CUBE_RAY;
		      cube_coord.c[1] =  CUBE_RAY-((float)(gy*RESOLUTION_W+y))*y_step;
		      cube_coord.c[2] =  CUBE_RAY-((float)(gx*RESOLUTION_H+x))*x_step;
		      break;

		    case 'E':
		      // East
		      cube_coord.c[0] =  CUBE_RAY;
		      cube_coord.c[1] =  CUBE_RAY-((float)(gy*RESOLUTION_W+y))*y_step;
		      cube_coord.c[2] =  CUBE_RAY-((float)(gx*RESOLUTION_H+x))*x_step;
		      break;

		    default:
		      fprintf( stderr, "Unknown plane %c - Abort\n", PLANE); 
		      exit (EXIT_FAILURE);
		   }
		 
		 Coord3DToPolar( cube_coord, &lat, &lon);	  
		 
		 unsigned int ox = (( lon + M_PI ) / ( 2.0f * M_PI ) * (float)ORTHO_RESOLUTION_W);
		 unsigned int oy = ORTHO_RESOLUTION_H - (( lat + M_PI/2.0f ) / ( M_PI) * (float)ORTHO_RESOLUTION_H);
		 	
		 if ( ox < ORTHO_RESOLUTION_W/2 )
		   {		      
		      cobe.SetPixelComp( x, y, Texture::B, ri_west.GetPixelComp( ox, oy, RawImage::R) );
		      cobe.SetPixelComp( x, y, Texture::G, ri_west.GetPixelComp( ox, oy, RawImage::G) );
		      cobe.SetPixelComp( x, y, Texture::R, ri_west.GetPixelComp( ox, oy, RawImage::B) );
		   }
		 else /* ox >= ORTHO_RESOLUTION_W && ox < ORTHO_RESOLUTION_W*2 */
		  {
		      cobe.SetPixelComp( x, y, Texture::B, ri_east.GetPixelComp( ox-ORTHO_RESOLUTION_W/2, oy, RawImage::R) );
		      cobe.SetPixelComp( x, y, Texture::G, ri_east.GetPixelComp( ox-ORTHO_RESOLUTION_W/2, oy, RawImage::G) );
		      cobe.SetPixelComp( x, y, Texture::R, ri_east.GetPixelComp( ox-ORTHO_RESOLUTION_W/2, oy, RawImage::B) );   
		   }
		 
	      }
	  
	  // Save destination Texture
	  Node node = Node( base_lod, gx, gy, Cube::GetFace(PLANE));
	  
	  char filename[128];
	  
	  sprintf( filename, "earth_cobe/%c/%04d.bmp", PLANE, node.GetNumber());
	  cobe.Save( filename);
              
	  fprintf( stdout, "\r%4d / %4d saved   ", gy*base_width+gx, base_width*base_width); 
	  fflush( stdout);
       }   
}
