#include <geotiff.h>
#include <xtiffio.h>
#include <geo_normalize.h>
#include <geovalues.h>
#include <tiffio.h>
#include <stdio.h>
#include <string.h>
#include "geoimage.hpp"

//#define BILINEAR 1

GeoImage::GeoImage()
{
   infos_ready = false;
   image_allocated = false;
   image_ready = false;
   filename = NULL;
}

GeoImage::~GeoImage()
{
   UnloadImage();
   if ( filename)
     delete[] filename; 
}
   
void GeoImage::LoadInfos( char *filename)
{
   TIFF *tif =NULL;  // TIFF-level descriptor
   GTIF	*gtif =NULL; // GeoKey-level descriptor

   this->filename = new char[strlen( filename)+1];
   strcpy( this->filename, filename);
   
   // Open the file
   
   tif = XTIFFOpen( filename, "r");
   
   if ( tif == NULL) 
     {
	fprintf( stderr, "GeoImage::LoadFromFile:: can't load %s\n", filename);
	return;
     }
   
   gtif = GTIFNew( tif);
   
   if ( gtif == NULL)
     {
        XTIFFClose( tif);
	fprintf( stderr, "GeoImage::LoadFromFile: failed in GTIFNew\n");
        return;
     }

   GTIFDefn *defn = new GTIFDefn;
   
   if( GTIFGetDefn( gtif, defn ) )
     {	
	GTIFPrintDefn( defn, stdout );
	
	fprintf( stderr, "PROJ.4 Definition: %s\n", GTIFGetProj4Defn(defn));
	
	TIFFGetField( tif, TIFFTAG_IMAGEWIDTH, &image_width );
	TIFFGetField( tif, TIFFTAG_IMAGELENGTH, &image_height );
	
	fprintf( stderr, "GeoImage: dimension image: %d %d\n", image_width, image_height);
	double x = 0;
	double y = image_height;
	
	// Try to transform the coordinate into PCS space
	if( GTIFImageToPCS( gtif, &x, &y ) )
	  {
	     fprintf( stderr, "GeoImage: so_longitude=%f so_latitude=%f\n", x, y);  
	     zone.SetRootLongitude( x);
	     zone.SetRootLatitude( y);
	  }
	
	x = image_width;
	y = 0.;
	if( GTIFImageToPCS( gtif, &x, &y ) )
	  {
	     zone.SetLongitudeWidth( x - zone.GetRootLongitude());
	     zone.SetLatitudeWidth(  y - zone.GetRootLatitude());
	     fprintf( stderr, "GeoImage: width=%f height=%f\n", zone.GetLongitudeWidth(), zone.GetLatitudeWidth());  
	     
	  }
	
	infos_ready =true;
     }
   
   delete defn;
   
   GTIFFree(gtif);
   XTIFFClose(tif);
}

unsigned int GeoImage::GetPixel( double lon, double lat)
{
   if ( image_ready)
     {
#ifdef BILINEAR
	unsigned int x1 = (unsigned int)(( lon - zone.GetRootLongitude() ) / zone.GetLongitudeWidth() * (double)image_width); 
	unsigned int y1 = (unsigned int)(( lat - zone.GetRootLatitude() ) / zone.GetLatitudeWidth() * (double)image_height); 
	unsigned int x2 = (unsigned int)(( lon - zone.GetRootLongitude() ) / zone.GetLongitudeWidth() * (double)image_width); 
	unsigned int y2 = (unsigned int)(( lat - zone.GetRootLatitude() ) / zone.GetLatitudeWidth() * (double)image_height); 
	unsigned int x3 = (unsigned int)(( lon - zone.GetRootLongitude() ) / zone.GetLongitudeWidth() * (double)image_width); 
	unsigned int y3 = (unsigned int)(( lat - zone.GetRootLatitude() ) / zone.GetLatitudeWidth() * (double)image_height); 
	unsigned int x4 = (unsigned int)(( lon - zone.GetRootLongitude() ) / zone.GetLongitudeWidth() * (double)image_width); 
	unsigned int y4 = (unsigned int)(( lat - zone.GetRootLatitude() ) / zone.GetLatitudeWidth() * (double)image_height); 
	double ax = 0.;
	double ay = 0.;
	
	return ax * image[]...;

#else // plus proche voisin
		     
	unsigned int x = (unsigned int)(( lon - zone.GetRootLongitude() ) / zone.GetLongitudeWidth() * (double)image_width); 
	unsigned int y = (unsigned int)(( lat - zone.GetRootLatitude() ) / zone.GetLatitudeWidth() * (double)image_height); 
	
	//fprintf( stderr, "get_pixel %d %d\n", x, y);
	return image[y*image_width+x];
#endif
     }
   else
     return 0;
}

unsigned int GeoImage::GetPixel( unsigned int x, unsigned y)
{
   if ( image_ready)
     {
	//fprintf( stderr, "get_pixel %d %d\n", x, y);
	return image[y*image_width+x];
     }
   else
     return 0;
}

void GeoImage::LoadImage()
{
   if ( !infos_ready)
     return;
   
   TIFF *tiff;
   unsigned int img_width, img_height;
   
   // turn off memory mapped access with m flag
   //   tiff = TIFFClientOpen( "GeoImage::LoadImage", "rm", (thandle_t)src, 
   //			 tiff_read, tiff_write, tiff_seek, tiff_close, tiff_size, NULL, NULL);
  
   tiff = TIFFOpen( filename, "r");
   
   if( !tiff)
     return;
   
   /* Retrieve the dimensions of the image from the TIFF tags */
   TIFFGetField(tiff, TIFFTAG_IMAGEWIDTH, &img_width);
   TIFFGetField(tiff, TIFFTAG_IMAGELENGTH, &img_height);

   image = new unsigned int[image_height*image_width];
   image_allocated = true;
   fprintf( stderr, "GeoImage::LoadImage: %dKo allocated\n", img_height*img_width*sizeof(unsigned int)/1024);

   if ( img_width != image_height && img_height != image_height)
     fprintf( stderr, "GeoImaghe::LoadImage: WARNING image size have changed since we read infos\n");
   
   if( TIFFReadRGBAImage( tiff, img_width, img_height, (uint32 *)image, 0))
     {  
	fprintf( stderr, "GeoImage::LoadImage:: successfull\n");
	image_ready = true;
     }
   
   TIFFClose(tiff);
}

void GeoImage::UnloadImage()
{
   if ( image_allocated)
     {
	image_allocated = false;
	image_ready = false;
	delete []image;
     }
}
