#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include <SDL.h>
#include <string.h>
using namespace std;
#include "../projection.hpp"
#include "../VectObjects.hpp"
#include "../VectInfos.hpp"
#include "../Node.hpp"
#include "../Grid.hpp"

struct City
{
   float latitude_deg;
   float longitude_deg;
   Vector3f position;
   
   bool is_capital;
   unsigned int population;
   string name;
   string country;
   unsigned int flag_id;
};

vector <VectInfo *> * load_cities( string filename)
{
   FILE *f;
   char line[1024];
   vector <VectInfo *> *vect_infos = new vector <VectInfo *>;
   
   f = fopen( filename.c_str(), "r");
   
   if ( f == NULL )
     {
	fprintf( stderr, "load_cities: can't open file %s for reading\n", filename.c_str());
	exit( EXIT_FAILURE);
     }
  
   while( !feof( f))
     {
	fgets( line, 1024, f);
	
	if ( line[0] != '#')
	  {
	     int lon_deg, lon_min, lat_deg, lat_min;
	     char lat_sign, lon_sign, capital_flag;
	     unsigned int population = 0;
	     char city_name[1024], country[1024];   
	      
	     City city;
	     
	     if ( sscanf( line, "\"%[^\"]\" %d %d %c %d %d %c %c \"%[^\"]\" %d", 
		     city_name, 
		     &lat_deg, &lat_min, &lat_sign,
		     &lon_deg, &lon_min, &lon_sign, &capital_flag, country, &population) > 8 ) 
	       {
		  VectObject *vo = new VO_Point(( ((float) lat_deg) + ((float) lat_min) / 60.0f ) * ((lat_sign=='N')?1.0f:-1.0f),( ((float) lon_deg) + ((float) lon_min) / 60.0f ) * ((lon_sign=='E')?1.0f:-1.0f));
		  VectProperties *vp = new VectProperties();
		  
		  //vp->AddString( "label", city_name);
		  //if ( capital_flag == 'Y')
		  //  vp->AddString( "Flag" , country); 
		 
		  //city.population = population;
		  //city.flag_id = flags_table->GetFlagNo( country);
	
		  VectInfo *vi = new VectInfo( vo, vp, "VO_POINT", "Cities");
		  
		  vect_infos->push_back( vi);
	       }
	  }	
     }
   
   fclose( f);

   fprintf( stderr, "cities loaded\n");

}

vector <VectInfo *> * load_pof( string filename)
{
   FILE *f;
   char *full_line = new char[2048+1];
   vector <VectInfo *> *vect_infos = new vector <VectInfo *>;
  
   f = fopen( filename.c_str(), "r");
   
   if ( f == NULL )
     {
	fprintf( stderr, "load_cities: can't open file %s for reading\n", filename.c_str());
	exit( EXIT_FAILURE);
     }
   
   while( !feof( f))
     {
	char *line = full_line;
	
	fgets( line, 2048, f);
	
	if ( line[0] != '#')
	  {
	     int RC; // Region code
	     int UFI; // Unique Future Identifier 
	     int UNI; // Unique Name Identifier
	     float LAT; // Latitude (deg)
	     float LONG; // Longitude (deg)
	     char DMS_LAT[64]; // WGS84 style latitude ( deg, min, sec)
	     char DMS_LONG[64]; // WGS84 style longitude ( deg, min, sec)
	     char UTM[4+1]; // Universal Transverse Mercato
	     char JOG[7+1]; // Joint Operations Graphic
	     char FC; // Feature Classification
	     char DSG[5+1]; // Feature Designation Code
	     //int PC; // Populated Place Classification
	     char PC[64]; // AGAINST WHAT IS WRITTEN IN REFERENCE 
	     char PCC[2+1]; // Primary Country Code
	     char ADM1[2+1]; // First-order administrative division
	     char ADM2[200+1]; // Second-order administrative division
	     int DIM; // Dimension
	     char CC2[2+1]; //Secondary Country Code
	     char NT; // Name Type
	     char LC[2+1]; // Language Code
	     char SHORT_FORM[128+1]; 
	     char GENERIC[128+1];
	     char SORT_NAME[200+1];
	     char FULL_NAME[200+1];
	     char FULL_NAME_ND[200+1];
	     char MOD_DATE[10+1];
	     
	     int nb_items;
	     
	     char *item;	     
	     
	     item = strsep( &line, "\t");
	     RC = item ? atoi( item) : 0;
	     item = strsep( &line, "\t");
	     UFI = item ? atoi( item) : 0; 
	     item = strsep( &line, "\t");
	     UNI = item ? atoi( item) : 0;
	     item = strsep( &line, "\t");
	     LAT = item ? atof( item) : 0.0f;
	     item = strsep( &line, "\t");
	     LONG = item ? atof( item) : 0.0f;
	     item = strsep( &line, "\t");
	     strncpy( DMS_LAT, item ? item : "", 64);
	     item = strsep( &line, "\t");
	     strncpy( DMS_LONG, item ? item : "", 64); 
	     item = strsep( &line, "\t");
	     strncpy( UTM, item ? item : "", 4); 
	     item = strsep( &line, "\t");
	     strncpy( JOG, item ? item : "", 7);   
	     item = strsep( &line, "\t"); 
	     FC = item ? item[0] : '?';
	     item = strsep( &line, "\t");
	     strncpy( DSG, item ? item : "", 5);
	     item = strsep( &line, "\t");
	     strncpy( PC, item ? item : "", 64); // AGAINST WHAT IS WRITTEN IN REFERENCE 
	     item = strsep( &line, "\t");
	     strncpy( PCC, item ? item : "", 2);
	     item = strsep( &line, "\t");
	     strncpy( ADM1, item ? item : "", 2);
	     item = strsep( &line, "\t");
	     strncpy( ADM2, item ? item : "", 200);
	     item = strsep( &line, "\t");
	     DIM = item ? atoi( item) : 0;
	     item = strsep( &line, "\t");
	     strncpy( CC2, item ? item : "", 2);
	     item = strsep( &line, "\t");
	     //NT = item[0]; //IGNORE
	     item = strsep( &line, "\t");
	     strncpy( LC, item ? item : "", 2);
	     item = strsep( &line, "\t");
	     strncpy( SHORT_FORM, item ? item : "", 28);
	     item = strsep( &line, "\t");
	     strncpy( GENERIC, item ? item : "", 200);
	     item = strsep( &line, "\t");
	     strncpy( SORT_NAME, item ? item : "", 200);
	     item = strsep( &line, "\t");
	     strncpy( FULL_NAME, item ? item : "", 200);
	     item = strsep( &line, "\t");
	     strncpy( FULL_NAME_ND, item ? item : "", 200);
	     item = strsep( &line, "\t");
	     strncpy( MOD_DATE, item ? item : "", 10);
	     item = strsep( &line, "\t");

	     if  ( FC == 'U')
	       {
		 fprintf( stdout, "[ %f %f ] %s ( %c  -> %s)\n", LAT, LONG, FULL_NAME_ND, FC, PC);
	     
		  VectObject *vo = new VO_Point( LAT, LONG);
		  VectProperties *vp = new VectProperties();
		  
		  //vp->AddString( "label", city_name);
		  //if ( capital_flag == 'Y')
		  //  vp->AddString( "Flag" , country); 
		 
		  //city.population = population;
		  //city.flag_id = flags_table->GetFlagNo( country);
	
		  VectInfo *vi = new VectInfo( vo, vp, "VO_POINT", "Cities");
		    
		  vect_infos->push_back( vi);
	       }
	  }	
     }
   
   fclose( f);
   
   fprintf( stderr, "%d pof loaded\n", vect_infos->size());

   return vect_infos;
}

  
int main( int argc, char **argv)
{
   int base_lod = 0;
   char plane = '0';
   string input_file;
   string extension;
   string dest_path;
   string data_name;
   bool print_statistics = false;
   
   /* Process parameters */

   for(int a=1;a<argc;a++) 
     {
	if ( strcmp( argv[a], "--help") == 0 ) 
	  {
	     fprintf( stdout, "usage: %s [options]\n\n", argv[0]) ;
	     fprintf( stdout, "options are:\n") ;
	     fprintf( stdout, "  --help               Display this help notice\n") ;
	     fprintf( stdout, "  --base-lod <lod>     Build level lod (default %d)\n", 0) ;
	     fprintf( stdout, "  --plane <id>         Select plane to build: N|S|F|B|E|W\n");
	     fprintf( stdout, "  --input-file <file>  Use this file for data source\n");
	     fprintf( stdout, "  --ext <ext>          Output file with this extension\n");
	     fprintf( stdout, "  --dest-path <path>   Path to store files\n");
	     fprintf( stdout, "  --data-name <name>   Name of vectorial data\n");
	     return EXIT_FAILURE;
	  }
	else if ( strcmp( argv[a], "--base-lod") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  base_lod = atol( argv[++a]);
	       }
	     else
	       {
		  fprintf( stderr, "--base-lod must be followed by an integer\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--plane") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  plane = argv[++a][0];
	       }
	     else
	       {
		  fprintf( stderr, "--plane must take a plane id\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--input-file") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  input_file = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--input-file must take a filename\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--ext") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  extension = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--ext must take a file extension\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--dest-path") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  dest_path = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--dest-path must take a path\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--data-name") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  data_name = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--data-name must take a name\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--print-statistics") == 0 )
	  {
	     print_statistics = true;
	  }	
	else
	  {
	     fprintf( stderr, "%s unknow parameter (--help)\n", argv[a]);
	  }
	
     }
   
   if ( plane == '0' || input_file == string("") || extension == string("") ||
	dest_path == string("") || data_name == string("") )
     {
	fprintf( stderr, "You need to specify: plane, input_file, extension, dest_path and data_name parameters\n");
	exit( EXIT_FAILURE);
     }
   
   Cube::Face face;
   
   try
     {
	face = Cube::GetFace( plane);
     }
   catch( string str)
     {
	cerr << str << endl;
	exit( EXIT_FAILURE);
     }
   
   float precision = 1.0f / 512.0f / ((float)(1>>base_lod)) *4.0f;
   
   // 1) Load Data
  
   vector< VectInfo *> *points_of_interest;

   points_of_interest = load_pof( input_file);
   
   
   // 2) Construct grids
     
   unsigned int base_width = 1 << base_lod;
   
   fprintf( stderr, "Will build the base (%dx%d) of a quadtree made of %d lod\n", base_width, base_width, base_lod);

   Grid< Cube::Coord> grid = Grid< Cube::Coord>( base_width+1, base_width+1);
   // 0 Front, 1 East, 2 Back, 3 West, 4 North, 5 South  
   
   for( unsigned int gy = 0 ; gy < base_width+1 ; gy++)
     for( unsigned int gx = 0 ; gx < base_width+1 ; gx++)
       {
	  float x = ((float)gx) * 2.0f * CUBE_RAY / (float) base_width;
	  float y = ((float)gy) * 2.0f * CUBE_RAY / (float) base_width;
	  
	  grid.Set( gx, gy, Cube::Coord( -1.0f+x,  -1.0f+y, face));
       }    
   
   fprintf( stderr, "Building plane %c\n", plane);
   
   // Sweep all base nodes
   
   for( unsigned int gy = 0 ; gy < base_width ; gy++)
     for( unsigned int gx = 0 ; gx < base_width ; gx++)
       {
	  
	  VectInfoCollection collection;

	  // Clip lines for current node
	   
	  Cube::Coord cube_coord_clip_min = Cube::Coord( grid.Get(gx,   gy  ));
	  Cube::Coord cube_coord_clip_max = Cube::Coord( grid.Get(gx+1, gy+1));

	  // Process all points of interest
		  	  
	  for( unsigned int pof = 0 ; pof < points_of_interest->size() ; pof++)
	    {

	       VO_Point *current_point = (VO_Point *) (*points_of_interest)[pof]->GetVectObject();
		    
	       if ( true && true )
		 { // Point of interest Inside and viewer enought near from it
		    
		    collection.AddVectInfo( (*points_of_interest)[pof]);
		 }
	       else 
		 {// Outside
		   
		    // do nothing
		 }
	       
	    }
       
	  // Save
	  char output_file[1024];
	  Node node = Node( base_lod, gx, base_width-1-gy, face);

	  sprintf( output_file, "%s/%c/%04d.%s", dest_path.c_str(), plane, node.GetNumber(), extension.c_str());

	  
	  SDL_RWops *w_ops = SDL_RWFromFile( output_file, "wb");
   
	  if ( w_ops == NULL)
	    {
	       cerr << argv[0] << ": can't save to " << output_file << endl;
	    }
	  else
	    {
	       collection.SaveBinary( w_ops);
	       SDL_RWclose( w_ops);
	       
	       fprintf( stdout, "\r%4d / %4d saved   ", gy*base_width+gx, base_width*base_width); 
	       fflush( stdout);
	    }
       }   
}



