#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include <SDL.h>
using namespace std;
#include "../projection.hpp"
#include "../VectObjects.hpp"
#include "../VectInfos.hpp"
#include "../Node.hpp"
#include "../Grid.hpp"

#define DEFAULT_BASE_LOD    0
#define DEFAULT_PLANE       'F'
#define DEFAULT_MAPGEN_FILE "Ressources/vecto/boundaries.txt"
#define DEFAULT_EXTENSION   "vbd" 
#define DEFAULT_DEST_PATH   "/tmp/blue_marble_bumped"
#define DEFAULT_DATA_NAME   "Boundaries"

bool clip_cube_coord( Cube::Coord cc_clip_min_xy, Cube::Coord cc_clip_max_xy, Cube::Coord cc_point)
{
   return cc_point.face == cc_clip_max_xy.face &&
          cc_point.c[0] <= cc_clip_max_xy.c[0] &&
          cc_point.c[1] <= cc_clip_max_xy.c[1] &&
          cc_point.c[0] >  cc_clip_min_xy.c[0] &&
          cc_point.c[1] >  cc_clip_min_xy.c[1];
}



int main( int argc, char **argv)
{
   int base_lod = DEFAULT_BASE_LOD;
   char PLANE = DEFAULT_PLANE;
   string mapgen_file = DEFAULT_MAPGEN_FILE;
   string extension = DEFAULT_EXTENSION;
   string dest_path = DEFAULT_DEST_PATH;
   string data_name = DEFAULT_DATA_NAME;
   bool print_statistics = false;
   
   /* Process parameters */

   for(int a=1;a<argc;a++) 
     {
	if ( strcmp( argv[a], "--help") == 0 ) 
	  {
	     fprintf( stdout, "usage: %s [options]\n\n", argv[0]) ;
	     fprintf( stdout, "options are:\n") ;
	     fprintf( stdout, "  --help               Display this help notice\n") ;
	     fprintf( stdout, "  --base-lod <lod>     Build level lod (default %d)\n", DEFAULT_PLANE) ;
	     fprintf( stdout, "  --plane <id>         Select plane to build: N|S|F|B|E|W\n");
	     fprintf( stdout, "  --mapgen-file <file> Use this file for data source\n");
	     fprintf( stdout, "  --ext <ext>          Output file with this extension\n");
	     fprintf( stdout, "  --dest-path <path>   Path to store files\n");
	     fprintf( stdout, "  --data-name <name>   Name of vectorial data\n");
	     fprintf( stdout, "  --print-statistics   Print statistics on standard output\n");
	     return EXIT_FAILURE;
	  }
	else if ( strcmp( argv[a], "--base-lod") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  base_lod = atol( argv[++a]);
	       }
	     else
	       {
		  fprintf( stderr, "--base-lod must be followed by an integer\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--plane") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  PLANE = argv[++a][0];
	       }
	     else
	       {
		  fprintf( stderr, "--plane must take a plane id\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--mapgen-file") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  mapgen_file = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--mapgen-file must take a filename\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--ext") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  extension = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--ext must take a file extension\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--dest-path") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  dest_path = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--dest-path must take a path\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--data-name") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  data_name = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--data-name must take a name\n");
		  return EXIT_FAILURE;
	       }
	  }
	else if ( strcmp( argv[a], "--print-statistics") == 0 )
	  {
	     print_statistics = true;
	  }
     }
   
   Cube::Face face;
   
   try
     {
	face = Cube::GetFace( PLANE);
     }
   catch( string str)
     {
	cerr << str << endl;
	exit( EXIT_FAILURE);
     }
   
   float precision = 1.0f / 512.0f / ((float)(1<<base_lod)) * 4.0f;
   
   // 1) Load Data
   
   VO_PolyPolyLine all_polypolylines;
   
   all_polypolylines.LoadFromTxtFile( mapgen_file);
      
   // 2) Construct grids
     
   unsigned int base_width = 1 << base_lod;
   
   fprintf( stderr, "Will build the base (%dx%d) of a quadtree made of %d lod\n", base_width, base_width, base_lod);

   Grid< Cube::Coord> grid = Grid< Cube::Coord>( base_width+1, base_width+1);
   // 0 Front, 1 East, 2 Back, 3 West, 4 North, 5 South  
   
   for( unsigned int gy = 0 ; gy < base_width+1 ; gy++)
     for( unsigned int gx = 0 ; gx < base_width+1 ; gx++)
       {
	  float x = ((float)gx) * 2.0f * CUBE_RAY / (float) base_width;
	  float y = ((float)gy) * 2.0f * CUBE_RAY / (float) base_width;
	  
	  grid.Set( gx, gy, Cube::Coord( -1.0f+x,  -1.0f+y, face));
       }    
   
   fprintf( stderr, "Building plane %c\n", PLANE);
   
   // Sweep all nodes
   
   for(unsigned int gy=0;gy<base_width;gy++)
     for(unsigned int gx=0;gx<base_width;gx++)
       {
	  // Clip all polypolylines for current node
	   
	  Cube::Coord cc_clip_min = Cube::Coord( grid.Get(gx,   gy  ));
	  Cube::Coord cc_clip_max = Cube::Coord( grid.Get(gx+1, gy+1));
	  VO_PolyPolyLine clipped_polypolyline;
	  
	  // Process all polylines

	  for( unsigned int pl = 0 ; pl < all_polypolylines.lines.size() ; pl++)
	    {
	       
	       VO_PolyLine *current_polyline = new VO_PolyLine();
	       bool previous_point_was_inside = false;
	       
	       if ( all_polypolylines.lines[pl]->points.size() < 2)
		 continue;
	       else if ( clip_cube_coord( cc_clip_min, cc_clip_max, all_polypolylines.lines[pl]->points[0].cube_coord))
		 {
		    current_polyline->AddPoint( all_polypolylines.lines[pl]->points[0]);
		    previous_point_was_inside = true;
		 }
	       	     
	       for( unsigned int p = 1 ; p < all_polypolylines.lines[pl]->points.size() ; p++ )
		 {
		    // Clip each points
		   
		    VO_Point current_point = all_polypolylines.lines[pl]->points[p];
		    		    
		    if ( clip_cube_coord( cc_clip_min, cc_clip_max, current_point.cube_coord))
		      { 
			 // point inside
			 
			 if ( previous_point_was_inside)
			   {
			      current_polyline->AddPoint( current_point);
			   }
			 else
			   {
			      // TODO: compute intersection from [ previous_point current_point ] and clipping segments  
			      current_polyline->AddPoint( all_polypolylines.lines[pl]->points[p-1]);
			      current_polyline->AddPoint( current_point);
			   }
			 previous_point_was_inside = true;
		      }
		    else 
		      {
			 // Outside
			 if ( previous_point_was_inside)
			   {
			      // TODO: compute intersection from [ previous_point current_point ] and clipping segments  
			      current_polyline->AddPoint( current_point);
			      clipped_polypolyline.AddPolyLine( current_polyline);
			      current_polyline = new VO_PolyLine();
			   }
			 else
			   ;// do nothing
			 
			 previous_point_was_inside = false;
		      
		      }
		 
		 } // points
	    
	       clipped_polypolyline.AddPolyLine( current_polyline);
			   
	    } // polylines
	  
	  // Resample
	  
	  VO_PolyPolyLine *resampled_polypolyline = clipped_polypolyline.Resample( precision);
	  
	  // Save
	  char filename[1024];
	  Node node = Node( base_lod, gx, base_width-1-gy, face);
	  	 
	  sprintf( filename, "%s/%c/%04d.%s", dest_path.c_str(), PLANE, node.GetNumber(), extension.c_str());
	  
	  SDL_RWops *w_ops = SDL_RWFromFile( filename, "wb");
	  if ( w_ops == NULL )
	    {
	       cerr << argv[0] << ": can't save to " << filename << endl;
	    }
	  else
	    {
	       VectProperties *dumy_properties = new VectProperties;
	       VectInfo vi( resampled_polypolyline, dumy_properties, "VO_POLYPOLYLINE", data_name);
	
	       vi.SaveBinary( w_ops);
	       SDL_RWclose( w_ops);
	       
	       if ( print_statistics)
		 {
		    unsigned int nb_lines = resampled_polypolyline->getNbLines();
		    unsigned int nb_points = resampled_polypolyline->getNbPoints();
		    fprintf( stdout, "[ %4d / %4d ] | %6d points | %6d lines\n", gy*base_width+gx, base_width*base_width, nb_points, nb_lines);
		 }
	       else
		 {
		    fprintf( stdout, "\r[ %4d / %4d ]  ", gy*base_width+gx, base_width*base_width); 
		    fflush( stdout);
		 }
	    }	  
	  //delete resampled_polypolyline;
       }   
}



