#include "VSDL.hpp"
#include <stdlib.h>

SDL_Window::SDL_Window( int width, int height, int flags, char* title) 
{
   videoFlags = SDL_ANYFORMAT|SDL_OPENGL ;
   if ( flags & FULLSCREEN )
     videoFlags |= SDL_FULLSCREEN ;
   if ( flags & ACCEL )
     videoFlags |= SDL_HWSURFACE ;
   if ( flags & DOUBLEBUF )
     videoFlags |=  SDL_DOUBLEBUF ;
   if ( flags & GL ) 
     videoFlags |= SDL_OPENGL ;
   if ( SDL_Init(SDL_INIT_VIDEO) < 0 ) {
      fprintf(stderr,"Couldn't init video\n") ;
      exit( -1) ;
   }

   //SDL_Init( SDL_INIT_EVERYTHING);
   
   //  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
 
   //   SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 32 );
   SDL_GL_SetAttribute( SDL_GL_BUFFER_SIZE, 24 );
    
   if ((screen=SDL_SetVideoMode( width, height, 0, videoFlags)) == NULL ) {
      fprintf(stderr, "SDL_Window: Couldn't set %dx%dxanybpp video mode: %s\n",
	      width,height,SDL_GetError());
      exit(-1) ;
   }
   printf("Set %dx%d at %d successfully.\n",
	  screen->w,
	  screen->h,
	  (int)screen->format->BitsPerPixel) ;
 
   
   int zbuf;
   SDL_GL_GetAttribute( SDL_GL_BUFFER_SIZE, &zbuf);
   fprintf( stdout, "ZBuffer sur %d bits\n", zbuf); 

   SDL_WM_SetCaption( title, title) ;
   videoReady = 1 ; 
}

SDL_Window::~SDL_Window() 
{  
   SDL_Quit() ; 
}

int SDL_Window::getWidth() 
{
   return screen->w ;
}

int SDL_Window::getHeight() 
{
   return screen->h ;
}

  
void SDL_Window::setColor( int r, int g, int b) 
{
   if ( videoReady != 1) 
     return ;
   currentColor = SDL_MapRGB( screen->format, r, g, b) ;   
}

void SDL_Window::setForeground( int r, int g, int b) 
{
   setColor( r, g, b) ; 
}

void SDL_Window::setBackground( int r, int g, int b) 
{
   if ( videoReady != 1) 
     return ;
   currentBackgroundColor = SDL_MapRGB( screen->format, r, g, b) ;   
}

void SDL_Window::clear() 
{
   SDL_Rect myRect ;
   
   if ( videoReady != 1 )
     return ;
   myRect.x = 0 ;
   myRect.y = 0 ;
   myRect.w = screen->w ;
   myRect.h = screen->h ;
   SDL_FillRect( screen, &myRect, currentBackgroundColor) ;
}

void SDL_Window::drawPoint( int x, int y) 
{
   if ( !_lockSurface() )
     return ;
   
   _drawPoint( x, y) ;
   _unLockSurface() ;
}

int SDL_Window::_lockSurface()
{
   if ( SDL_MUSTLOCK(screen) ) 
     return  SDL_LockSurface(screen) >= 0 ;
   else
     return 1 ;
}

void SDL_Window::_unLockSurface()
{
   if ( SDL_MUSTLOCK(screen) ) 
     SDL_UnlockSurface(screen) ;
}


void SDL_Window::_drawPoint( int x, int y) 
{
   Uint8 bpp ;
   
   if ( videoReady != 1)
     return ;
   
   if ( x >= 0 && x < screen->w-1 && y >= 0 && y < screen->h-1 ) {
      
      bpp = screen->format->BytesPerPixel ;
      switch( bpp) {
       case 2: {
	  *((Uint16 *)(((Uint8 *)screen->pixels)+y*screen->pitch+x*bpp)) = (Uint16)currentColor ;
       } break ;
       case 3: { /* Format/endian independent */
	  Uint8 r, g, b;
	  r = (currentColor>>screen->format->Rshift)&0xFF ;
	  g = (currentColor>>screen->format->Gshift)&0xFF ;
	  b = (currentColor>>screen->format->Bshift)&0xFF ;
	  *((((Uint8 *)screen->pixels)+y*screen->pitch+x*bpp)+screen->format->Rshift/8) = r ;
	  *((((Uint8 *)screen->pixels)+y*screen->pitch+x*bpp)+screen->format->Gshift/8) = g ;
	  *((((Uint8 *)screen->pixels)+y*screen->pitch+x*bpp)+screen->format->Bshift/8) = b ;
       } break ;
       case 4: {
	  *((Uint32 *)(((Uint8 *)screen->pixels)+y*screen->pitch+x*bpp)) = (Uint32)currentColor ;
       } break ;
      }
   }   
}

void SDL_Window::drawLine( int x1, int y1, int x2, int y2) 
{
   int Xdiff, Ydiff, diff ;
   float x, y, stepX, stepY ;
   int i ;
   
   if ( videoReady != 1) 
     return ;
   
   Xdiff = x2 - x1 ;
   Ydiff = y2 - y1 ;
   
   if ( abs(Xdiff) >= abs(Ydiff) ) { /* draw by the x */ 
      if ( Xdiff > 0 ) {
	 diff = Xdiff ;
	 x = x1 ;
	 y = y1 ;
      }
      else {
	 diff = -Xdiff ;
	 x = x2 ;
	 y = y2 ;
      }
      stepX = 1 ;
      stepY = (float)Ydiff / (float)Xdiff ;
   }
   else { /* else draw by the y */
      if ( Ydiff > 0 ) {
	 diff = Ydiff ;
	 x = x1 ;
	 y = y1 ;
      }
      else {
	 diff = -Ydiff ;
	 x = x2 ;
	 y = y2 ;
      }
      stepX = (float)Xdiff / (float)Ydiff ;
      stepY = 1 ;
   }
   
   if ( !_lockSurface() )
     return ;
   
   for(i=0;i<diff;i++) {
      x = x + stepX ;
      y = y + stepY ;
      _drawPoint( (int)x, (int)y) ;
   }
   _unLockSurface() ;
}

void SDL_Window::drawVerticalLine( int x, int y1, int y2)
{
   if ( !_lockSurface() )
     return ;
     
   if ( y2 > y1 ) 
     for( int i=y1;i<=y2;i++)
       _drawPoint( x, i) ;	 
   else
     for( int i=y2;i<=y1;i++)
       _drawPoint( x, i) ;	 
     
   _unLockSurface() ;
}


void SDL_Window::drawRect( int x, int y, int w, int h) 
{
   SDL_Rect myRect ;
   
   if ( videoReady != 1 )
     return ;
   
   myRect.x = x ;
   myRect.y = y ;
   myRect.w = w ;
   myRect.h = h ;
   SDL_FillRect( screen, &myRect, currentColor) ;
}

void SDL_Window::refresh() {
   
   if ( videoReady != 1 )
     return ; 
   SDL_Flip( screen) ;
}

int SDL_Window::waitFrame( int fps) {
   static Uint32 next_tick = 0 ;
   Uint32 this_tick ;
   int realfps ;
   
   this_tick = SDL_GetTicks() ;
   if ( this_tick <= next_tick ) {
      SDL_Delay( next_tick-this_tick) ;
      realfps = fps ;
   }
   else 
     realfps = 1000/(1000/fps+this_tick-next_tick) ;
   
   next_tick = SDL_GetTicks() + (1000/fps) ; 
   return( realfps) ; /* return the real fps */
}

int SDL_Window::pollEvent( SDL_Event* event)
{
   return SDL_PollEvent( event) ;
}

int SDL_Window::getKey() 
{
   SDL_Event event ;
   if ( SDL_PeepEvents( &event, 1, SDL_GETEVENT, SDL_KEYDOWN) )
     return event.key.keysym.sym ;
   else
     return -1 ;
}

int SDL_Window::getMouseEvents( SDL_Event* event) 
{
   if ( SDL_PeepEvents( event, 1, SDL_GETEVENT, SDL_MOUSEMOTION|SDL_MOUSEBUTTONDOWN|SDL_MOUSEBUTTONUP) )
     return 1 ;
   else
     return 0 ;
}

void SDL_Window::getMouseCoords( int *x, int *y)
{
   SDL_GetMouseState( x, y) ;
}

/*int SDL_Window::getJoystickEvents( SDL_Event* event) 
{   
   if ( SDL_PeepEvents( event, 1, SDL_GETEVENT, SDL_JOYAXISMOTION|SDL_JOYBALLMOTION|SDL_JOYHATMOTION|SDL_JOYBUTTONDOWN|SDL_JOYBUTTONUP) )
     1 ;
   else
     0 ;
}*/

int SDL_Window::isButtonRelease() 
{
   Uint8* keyMap = SDL_GetKeyState(NULL) ;
   return keyMap[SDLK_q] ;
}

void SDL_Window::showCursor( bool mode) 
{
   SDL_ShowCursor( mode?SDL_ENABLE:SDL_DISABLE) ;
}

void SDL_Window::toggleFullScreen()
{
   SDL_WM_ToggleFullScreen( screen) ;
}
