#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include "config.h"
#include "projection.hpp"

using namespace std;

#define DEFAULT_RADIUS 0.5L

//char Cube::FacePrefix[6] = { 'F', 'E', 'B', 'W', 'N', 'S'};

char Cube::FacePrefix[6] = { 'f', 'e', 'b', 'w', 'n', 's'};

Cube::Face Cube::GetFace( char face)
{
   switch( face)
     {
      case 'F': return Front;
      case 'E': return East;
      case 'B': return Back;
      case 'W': return West;
      case 'N': return North;
      case 'S': return South;
      default: throw string( "GeatFace: Unknown face prefix");
     }
   return Front;
}
   
char Cube::GetFace( Face face)
{
   return FacePrefix[face];
}


// Given 2 angles ( latitude and longitude ) and the sphere radius,
// this function returns the 3D coordinates of the corresponding point 
// on the surface of the sphere ( centered on origin (0,0,0) )

Coord3D PolarToCoord3D( tpv_float lat, tpv_float lon, tpv_float ray)
{
   Coord3D c;
   
   c.c[0] = ray * sin(lon/180.0f*M_PI) * cos(lat/180.0f*M_PI);
   c.c[1] = ray * sin(lat/180.0f*M_PI);
   c.c[2] = ray * cos(lon/180.0f*M_PI) * cos(lat/180.0f*M_PI);
   
   return c;
}

Vector3f PolarToCoord3f( tpv_float lat, tpv_float lon, tpv_float radius)
{
   Vector3f p;
   
   p[0] = radius * sin(lon/180.0f*M_PI) * cos(lat/180.0f*M_PI);
   p[1] = radius * sin(lat/180.0f*M_PI);
   p[2] = radius * cos(lon/180.0f*M_PI) * cos(lat/180.0f*M_PI);
   
   return p;
}

// Convert 3D Coordinates to angles
// Return angles between given coordinate and origin O(0,0,0)
// lat is angle between X and Z axis 
// lon is angle betwen  Y and XZ axis
// 
// this function is used to find sphere lat/long from the inscribed cube faces coordinates
// when the shpere is projected on a cube

void Coord3DToPolar( Coord3D point, tpv_float *lat, tpv_float *lon)
{
   *lon = atan2( point.c[0], point.c[2]);
   *lat = atan2( point.c[1], sqrt( point.c[0]*point.c[0] + point.c[2]*point.c[2]));
}

void Coord3fToPolar( Vector3f point, tpv_float *lat, tpv_float *lon)
{
   *lon = atan2( point[0], point[2]);
   *lat = atan2( point[1], sqrt( point[0]*point[0] + point[2]*point[2]));
}

Coord3D CubeToSphere( Coord3D c_point)
{
   tpv_float lat, lon;
   Coord3DToPolar( c_point, &lat, &lon);
   
   return PolarToCoord3D( lat/M_PI*180.0f, lon/M_PI*180.0f);
}

Vector3f CubeToSphere3f( Vector3f c_point)
{
   float lat, lon;
   Coord3fToPolar( c_point, &lat, &lon);
   
   return PolarToCoord3f( lat/M_PI*180.0f, lon/M_PI*180.0f, 0.5f);
}

// Return

void PolarToFaceCoord( tpv_float a1, tpv_float a2, tpv_float face_half_width, tpv_float *x, tpv_float *y)
{
   *x = tan( a2/180.0f*M_PI)*face_half_width;
   
   float PX = sqrt( *x**x + face_half_width*face_half_width); 
   
   *y = tan( a1/180.0f*M_PI) * PX;
}

// 

#define CUBE_RAY (1.0f)

Cube::Coord PolarToCubeCoord( tpv_float lat_deg, tpv_float lon_deg)
{
   // test are needed to find the face of the cube on wihch
   // the point will be projected on
   // Note: since the cube is insribed into the sphere, there is no ambiguity
   // between cube's face for any projected point

   Coord3D P = PolarToCoord3D( lat_deg, lon_deg, 1.0f);
   
   tpv_float xy = P.c[0]/P.c[1];
   tpv_float xz = P.c[0]/P.c[2];
   tpv_float yx = P.c[1]/P.c[0];
   tpv_float yz = P.c[1]/P.c[2];
   tpv_float zx = P.c[2]/P.c[0];
   tpv_float zy = P.c[2]/P.c[1];

   Cube::Coord cube_coord;
   
   // make lon_deg to be between [ 0 ; 360 [
   while( lon_deg < 0.0f)
     lon_deg += 360.0f;
   while( lon_deg >= 360.0f)
     lon_deg -= 360.0f;
   
   if ( lon_deg >= 315.0f || lon_deg < 45.0f ) // should be projected on ( front,north,south)
     {
	cube_coord.c[0] =  xz * CUBE_RAY;
	cube_coord.c[1] =  yz * CUBE_RAY;
	cube_coord.face = Cube::Front;
     }
   else if ( lon_deg >= 45.0f && lon_deg < 135.0f) // should be projected on ( east,north,south)
     {
	cube_coord.c[0] = -zx * CUBE_RAY;
	cube_coord.c[1] =  yx * CUBE_RAY;
	cube_coord.face = Cube::East;   
     }
   else if ( lon_deg >= 135.0f && lon_deg < 225.0f) // should be projected on ( back,north,south)
     {
	cube_coord.c[0] = -xz * -CUBE_RAY;
	cube_coord.c[1] =  yz * -CUBE_RAY;
	cube_coord.face = Cube::Back;
     }
   else // should be projected on ( west,north,south)
     {
	cube_coord.c[0] =  zx * -CUBE_RAY;
	cube_coord.c[1] =  yx * -CUBE_RAY;
	cube_coord.face = Cube::West;
     }

   if ( cube_coord.c[1] > CUBE_RAY) // point is projected on north
     {
	cube_coord.c[0] =  xy * CUBE_RAY;
	cube_coord.c[1] = -zy * CUBE_RAY;
	cube_coord.face = Cube::North;
     }
   else if ( cube_coord.c[1] < -CUBE_RAY) // point is projected on south
     {
	cube_coord.c[0] = -xy * -CUBE_RAY;
	cube_coord.c[1] =  zy * -CUBE_RAY;
	cube_coord.face = Cube::South;
     }
   
   
   /*
   //
   
   int face = 0 ;
   float on_cube_face_x;
   float on_cube_face_y;
   float on_cube_face_z;
   
   if ( lon_deg < 0 ) // the point comes from western emisphere
     {
	fprintf( stderr, "projection on west emisphere not supported yet\n");	
     }
   
   else // the point comes from eastern emisphere
     { 

	// so project the point on the east face 
	
	lon_deg -= 90.0f;
	
	PolarToFaceCoord( lat_deg, lon_deg, CUBE_RAY, &on_cube_face_z, &on_cube_face_y);
	
	if ( on_cube_face_y > CUBE_RAY) // the point is projected on north face
	  {
	     float tmp = lat_deg;
	     lat_deg = lon_deg;
	     lon_deg = tmp;
	     PolarToFaceCoord( lat_deg, lon_deg, CUBE_RAY, &on_cube_face_x, &on_cube_face_z);
	     on_cube_face_y = CUBE_RAY;
	     face = 0;
	     fprintf( stderr, "\nnorth\n");
	  }
	else if ( on_cube_face_y < - CUBE_RAY) // the point is projected on south face
	  {
	     on_cube_face_y = -CUBE_RAY;
	     face = 1;
	     fprintf( stderr, "\nsouth\n");
	  }
	else if ( on_cube_face_z < -CUBE_RAY ) // the point is projected on front face  
	  {	     
	     lon_deg -= 90.0f;
	     PolarToFaceCoord( lat_deg, lon_deg, CUBE_RAY, &on_cube_face_x, &on_cube_face_y);
	     on_cube_face_z = -CUBE_RAY;	   
	     face = 2;
	     fprintf( stderr, "\nfront\n");
	  }	
	else if ( on_cube_face_z > CUBE_RAY ) // the point is projected on back face  
	  {
	     lon_deg += 90.0f;
	     lon_deg = 180 - lon_deg;
	     PolarToFaceCoord( lat_deg, lon_deg, CUBE_RAY, &on_cube_face_x, &on_cube_face_y);
	     on_cube_face_z = +CUBE_RAY;
	     face = 3;
	     fprintf( stderr, "\nback\n");
	  }
	else // the point is projected on east face
	  {
	     on_cube_face_x = CUBE_RAY; 
	     face = 4;
	     fprintf( stderr, "\neast\n");
	  }
     }
   */
//   return Coord3D( on_cube_face_x, on_cube_face_y, -on_cube_face_z);

   return cube_coord;
}

using namespace Cube;

Coord3D Coord::GetCoord3D( float cube_size)
{
   Coord3D c;
  
   cube_size = 1.0f;
   
   
   switch( face)
     {
      case Front:
	c.c[0] =  Coord::c[0];
	c.c[1] =  Coord::c[1];
	c.c[2] =  cube_size;
	break;
      case East:
	c.c[0] =  cube_size;
	c.c[1] =  Coord::c[1];
	c.c[2] = -Coord::c[0];
	break;
      case Back:
	c.c[0] = -Coord::c[0];
	c.c[1] =  Coord::c[1];
	c.c[2] = -cube_size;	
	break;
      case West:
	c.c[0] = -cube_size;
	c.c[1] =  Coord::c[1];
	c.c[2] =  Coord::c[0];
	break;
      case North:
	c.c[0] =  Coord::c[0];
	c.c[1] =  cube_size;
	c.c[2] = -Coord::c[1];
	break;
      case South: 
	c.c[0] =  -Coord::c[0];
	c.c[1] =  -cube_size;
	c.c[2] =   Coord::c[1];
	break;
     }
      
   return c;
}

/*
void Coord3D::Normalize()
{
   tpv_float norm = sqrt( c[0]*c[0] + c[1]*c[1] + c[2]*c[2]) / 2.0f;
   
   if ( norm != 0.0f)
     {
	c[0] /= norm;
	c[1] /= norm;
	c[2] /= norm;
     }
   else
     c[0]=c[1]=c[2]=0.0f;
}
*/
