/* 
  -+-------------------------------------------------------------------------+-
    LATE, Look At The Earth - Copyright(C) 2004, Jean-Christophe Duberga
   
    File: main.cpp
   
    Purpose: Initialise SDL and OpenGL 
             Collect and treat events
             Manage the high level Planet object
  -+-------------------------------------------------------------------------+-
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Jean-Christophe Duberga, 
    jeanchristophe.duber@free.fr
    10, Rue de Solesse
    33290 BLANQUEFORT
    FRANCE
  
  -+-------------------------------------------------------------------------+-
*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "gl_stuff.hpp"
#include <GL/glu.h>
#include "sdl_stuff.hpp"
#include "config.h"
#include "projection.hpp"
#include "Planet.hpp"
#include "Camera.hpp"
#include "Observer.hpp"
#include "CloudsLayer.hpp"
#include "Options.hpp"
#include "Cities.hpp"
#include "math/Vector.hpp"
#ifdef _USE_GUI
# include "SideBar.hpp"
#endif /* USE_GUI */

#define MINIMAL_DISTANCE  0.52f
#define MAXIMAL_DISTANCE  20.0f
#define INITIAL_DISTANCE  1.0f
#define FPS_LIMIT         1000
#define DEFAULT_DATASET   "blue_marble_bumped.pkg"
#define EARTH_RADIUS_KM   (12742.6f/2.0f) // values found from www.wikipedia.org
#define MOON_RADIUS_KM    (3474.8f/2.0f) 
#define MOON_ORBITAL_RADIUS_KM (384400.0f/10.0f)
#define CAMERA_NEAR       0.001f
#define CAMERA_FAR        50.0f
#define CAMERA_FOV        60.0f

/* Global variables -> should be removed before 1.0 release */

#ifdef _USE_GUI
aedApp app;
#endif

/* end of global variables */


void lighting( bool light)
{
   if ( light)
     {
	glEnable(GL_LIGHTING);
	glShadeModel(GL_SMOOTH);
	
	// Set up ambiant
	GLfloat ambiant_light[] = { 0.1, 0.1, 0.1, 1.0};
	glLightModelfv( GL_LIGHT_MODEL_AMBIENT, ambiant_light );
	glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL,GL_SEPARATE_SPECULAR_COLOR);
	
	// Set up lighting.
	float light0_ambient[4]  = { 0.1, 0.1, 0.1, 0.1 };
	float light0_diffuse[4]  = { 1.0, 1.0, 1.0, 1.0 };
	float light0_specular[4] = { 1.0, 1.0, 1.0, 1.0 };
	float light0_position[4] = { 150000000.0f/(EARTH_RADIUS_KM*2.0f) /*1000.0*/, 500.0f, 0.0, 1.0 };
	glLightfv(GL_LIGHT0, GL_AMBIENT,  light0_ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE,  light0_diffuse);
	glLightfv(GL_LIGHT0, GL_SPECULAR, light0_specular);
	glLightfv(GL_LIGHT0, GL_POSITION, light0_position);
	glEnable(GL_LIGHT0);
		
	float mat_emission[4] = { 0.0f,  0.0f,  0.0f, 0.0f };
	float mat_ambient[4]  = { 0.85f, 0.85f, 1.0f, 0.1f };
	float mat_diffuse[4]  = { 0.85f, 0.85f, 1.0f, 0.1f };
	float mat_specular[4] = { 0.5f,  0.5f,  0.55f,1.0f };
	
	glColorMaterial( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glEnable( GL_COLOR_MATERIAL);
	
	glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, mat_emission);
	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT,  mat_ambient);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,  mat_diffuse);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
	glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, 25.0f);
     }
   else
     {
	glDisable( GL_LIGHTING);
     }
}

int main_( int argc, char *argv[])
{
   /* Process parameters */
   
   int width = 800;
   int height = 600;
   int sdl_video_flags = 0;
   char *dataset_file = DEFAULT_DATASET;
   
   for( int a = 1 ; a < argc ; a++) 
     {
	if ( strcmp( argv[a], "--help") == 0 ) 
	  {
	     fprintf( stderr, "usage: %s [options]\n\n", argv[0]) ;
	     fprintf( stderr, "options are:\n") ;
	     fprintf( stderr, "  --help               Display this help notice\n") ;
	     fprintf( stderr, "  --fullscreen         Fullscreen mode (default: no)\n") ;
	     fprintf( stderr, "  --datapath           Set base data path\n") ;
	     fprintf( stderr, "  --resolution w h     Set resolution (default %d %d)\n", width, height);
	     fprintf( stderr, "  --dataset <ds.pkg>   Load this dataset (default %s)\n", DEFAULT_DATASET);
	     
	     exit( EXIT_FAILURE) ;
	  }
	else if ( strcmp( argv[a], "--fullscreen") == 0 )
	  sdl_video_flags |= SDL_FULLSCREEN;
	else if ( strcmp( argv[a], "--resolution") == 0 )
	  {
	     if ( a+2 < argc )
	       {
		  width = options.screen_width = atoi( argv[++a]);
		  height = options.screen_height = atoi( argv[++a]);
	       }
	     else
	       {
		  fprintf( stderr, "--dataset must take a file parameter\n");
		  return EXIT_FAILURE;
	       }    
	  }
	else if ( strcmp( argv[a], "--dataset") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  dataset_file = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--dataset must take a file parameter\n");
		  return EXIT_FAILURE;
	       }    
	  }
	else
	  fprintf( stderr, "%s: unknow parameter %s\n", argv[0], argv[a]);
     }
   
   SDL_Surface *screen;
   char *title = "TinyPlanetViewer-0.6 (alpha)";
  
   // Init screen or window with SDL

   if ( SDL_Init( SDL_INIT_VIDEO) < 0 ) 
     {
	fprintf( stderr, "%s: can't init video!\n", argv[0]);
	return EXIT_FAILURE;
     }
    
   SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1);

   SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 24);

   if ( ( screen = SDL_SetVideoMode( width, height, 0, SDL_OPENGL|sdl_video_flags)) == NULL ) 
     {
	fprintf( stderr, "%s: can't set %dx%dxanybpp video mode: %s\n",
		 argv[0], width, height, SDL_GetError());
	return EXIT_FAILURE;
     }
   
   SDL_WM_SetCaption( title, title) ;

   SDL_ShowCursor( 1);
   
   SDL_EnableKeyRepeat( SDL_DEFAULT_REPEAT_DELAY,  SDL_DEFAULT_REPEAT_INTERVAL);

   // Setup OpenGL
   
   //gl_load_extensions();
   extgl_Initialize();
   glViewport( 0, 0, screen->w, screen->h);
   gl_init_garbage();   
   glDepthFunc(GL_LEQUAL);
   glPolygonMode( GL_FRONT_AND_BACK, GL_FILL);
   glEnable( GL_POLYGON_OFFSET_LINE);
   glPolygonOffset( 1.0f, 5.0f);
   glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE);
   glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
   glEnable( GL_NORMALIZE);// TODO normalize in precomputation
      
   Camera camera;
   camera.SetupProjectionMatrix( options.fov_deg, CAMERA_NEAR, CAMERA_FAR, width, height);
   Camera locked_camera;
   Observer_Ortho observer_ortho( MINIMAL_DISTANCE, MAXIMAL_DISTANCE, INITIAL_DISTANCE, 0.0f, 0.0f);
   Observer_Quaternion observer_quaternion;
   Observer_SpaceShip observer_spaceship( MINIMAL_DISTANCE, MAXIMAL_DISTANCE, Vector3f(0.0f, 0.0f, -2.0f));
   Observer *observer = &observer_ortho;
   Planet *p = new Planet();
   p->Initialize( dataset_file);
   Cities *cities = new Cities();
   CloudsLayer *clouds = new CloudsLayer();
   
#ifdef _USE_GUI
   
   app.setDefaultFontName( "Vera.ttf");
   SideBar *side_bar = new SideBar();

#endif /* USE_GUI */
   
   // Main loop
    
   while(1)   
     {
	
	// Poll and treat events
	
	SDL_Event event;
	
#ifndef _USE_GUI	
	while( SDL_PollEvent( &event) ) 
#else
	while( side_bar->desktop->pollEvent( &event)) 
#endif
	 {      
	     if ( (event.type == SDL_QUIT) || 
		  (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_ESCAPE)) 
	       {
		  delete p;
		  gl_quit_garbage();
		  SDL_Quit();
		  return EXIT_SUCCESS;
	       }
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F1)
	      options.draw_cities = !options.draw_cities;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F2)
	      options.draw_boundaries = !options.draw_boundaries;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F3)
	      options.draw_coastlines = !options.draw_coastlines;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F4)
	      options.draw_rivers = !options.draw_rivers;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F5)
	      options.draw_latitudes = !options.draw_latitudes;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F6)
	      options.draw_longitudes = !options.draw_longitudes;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F7)
	      options.lighting = !options.lighting;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F8)
	      options.draw_clouds = !options.draw_clouds;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F9)
	      {
		 observer = &observer_ortho;
		 SDL_ShowCursor( 1);
	      }
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F10)
	      {
		 observer = &observer_quaternion;
		 SDL_ShowCursor( 1);
	      }
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_F11)
	      {
		 observer = &observer_spaceship;
		 SDL_ShowCursor( 0);
	      }
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_f)
	       SDL_WM_ToggleFullScreen( screen);
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_p)
	      {
		 options.fov_deg += 1.0f;
		 camera.SetupProjectionMatrix( options.fov_deg, CAMERA_NEAR, CAMERA_FAR, width, height);
		 cout << "FOV: " << options.fov_deg << "" << endl;
	      }
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_m)
	      {
		 options.fov_deg -= 1.0f;
		 camera.SetupProjectionMatrix( options.fov_deg, CAMERA_NEAR, CAMERA_FAR, width, height);		 
		 cout << "FOV: " << options.fov_deg << "" << endl;
	      }
	    
#ifdef _DEBUG
	     
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_w)
	      options.wireframe = !options.wireframe;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_b)
	      options.bounding_box = !options.bounding_box;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_s)
	      options.subdivisions = !options.subdivisions;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_v)
	      options.debug_vecto = !options.debug_vecto;
	    if ( event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_l)
	      {
		 options.camera_locked = !options.camera_locked;
		 observer->UpdateCamera( &locked_camera);
		 locked_camera.GlTransform(); // to update frustum
	      }
	    
#endif /* _DEBUG */
	     
	    observer->OnSDLEvent( &event);
	     
	  }

	observer->Update();
	
	// Clear screen and setup OpenGL states
	
	glClearColor( 0.0f, 0.0f, 0.0f, 0.0f); // clear with black
	glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
	glEnable( GL_DEPTH_TEST);
	
	if (options.backface_culling)
	  {
	     glEnable( GL_CULL_FACE);
	     glFrontFace( GL_CCW);
	  }
	else
	  {
	     glDisable( GL_CULL_FACE);
	  }
			
	observer->UpdateCamera( &camera);
	camera.GlTransform();
	
#ifdef _DEBUG
	
	glPolygonMode( GL_FRONT_AND_BACK, options.wireframe?GL_LINE:GL_FILL); 
	
	if ( options.draw_rep )
	  {
	     gl_draw_repere( 1.0f);
	  }
	
	if ( options.draw_cube)
	  {
	     gl_draw_cube( CUBE_RAY);
	  }
	
#endif /* _DEBUG */
	     
	if ( options.draw_latitudes)
	  {
	     // Draw latitudes in blue ( every 10 degrees)
	     
	     glColor3f( 0.0f, 0.0f, 1.0f);
	     
	     for( int lat_deg =- 90 ; lat_deg < 90 ; lat_deg += 10)
	       p->DrawLatitude( lat_deg);
	  }
	
	if ( options.draw_longitudes)
	  {
	     // Draw longitudes in red ( every 10 degrees)
	     
	     glColor3f( 1.0f, 0.0f, 0.0f);
	     
	     for ( float lon_deg = 0.0f ; lon_deg < 360.0f ; lon_deg += 10.0f)
	       p->DrawMeridian( lon_deg);
	  }
	
	lighting( options.lighting);
	
#ifdef _DEBUG
	
	p->Draw( options.camera_locked ? &locked_camera : &camera);
	
#else /* _DEBUG not defined */

	p->Draw( &camera);
	
#endif
	
	GLint mx, my;
	float lat_deg, lon_deg;
	
	SDL_GetMouseState( &mx, &my);

	p->PickCoordinates( mx, my, &lat_deg, &lon_deg);

	if ( options.draw_cities)
	  cities->Draw( &camera);
	
	if ( options.draw_clouds)
	  {
	     float mat_spec[] = { 0.0f, 0.0f, 0.0f, 0.0f};
	     glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_spec);
	     glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, 0.0f);

	     clouds->Draw();
	  }
	
	Vector3f moon_position = PolarToCoord3f( 5.0f, 135.0f, MOON_ORBITAL_RADIUS_KM/(2.0f*EARTH_RADIUS_KM));
	glPushMatrix();
	glTranslatef( moon_position[0], moon_position[1], moon_position[2]);
	GLUquadricObj *sphereObj = gluNewQuadric();
	gluSphere( sphereObj, MOON_RADIUS_KM/(2.0f*EARTH_RADIUS_KM), 32, 32);
	glPopMatrix();
	
#ifdef _USE_GUI
	
	side_bar->SetLatitude( lat_deg);
	side_bar->SetLongitude( lon_deg);
	
	side_bar->Update();
	side_bar->Draw();	
	
#endif /* USE_GUI */

	SDL_GL_SwapBuffers();
	
	gl_clear_garbage();
	
	int fps = waitFrame( FPS_LIMIT);
	
#ifdef _DEBUG

	float altitude_km = (Vector3f( 0.0f, 0.0f, 0.0f).distance( camera.GetEyePosition()) - 0.5f ) * 2.0f * EARTH_RADIUS_KM;
	fprintf( stderr, "\rdrawn nodes: %2d/%3d (%3d fps) | cursor: %+3.2f %+3.2f | altitude: %4.1f km   ", p->GetDrawnNodes(), p->GetExploredNodes(), fps, lat_deg, lon_deg, altitude_km);

#endif /* _DEBUG */
	
     }   
}


int main( int argc, char *argv[])
{

   try
     {
	main_( argc, argv);
     }
   catch( std::string e)
     {
	cerr << argv[0] << " Aborted due to this uncaught exception: " << e << endl;
     }
}
