#include <SDL.h>
#include <SDL_endian.h>
#include <iostream>
#include <GL/gl.h>
#include "VectObjects.hpp"
#include "math/Vector.hpp"
#include "Options.hpp"

// TODO: make the read and write functions portable (taking care of endianess)

float debugging_colors[3*4] = 
{ 
   1.0f, 0.0f, 0.0f, 1.0f,
   0.0f, 1.0f, 0.0f, 1.0f,
   0.0f, 0.0f, 1.0f, 1.0f,
};

/* === VO_Point implemetation ======================================================== */

VO_Point::VO_Point( float lat_deg, float lon_deg, float radius)
{
   this->lat_deg = lat_deg;
   this->lon_deg = lon_deg;
   coord = PolarToCoord3f( lat_deg, lon_deg, radius);
   cube_coord = PolarToCubeCoord( lat_deg, lon_deg);
}

VO_Point::~VO_Point()
{
}

void VO_Point::LoadASCII( ifstream *r_stream)
{
   try 
     {
	*r_stream >> lon_deg;
	*r_stream >> lat_deg;
     }
   catch(...)
     {
	throw string( "VO_Point: (LoadASCII) can't read coordinates");	
     }     
   
   coord = PolarToCoord3f( lat_deg, lon_deg, RADIUS);
   cube_coord = PolarToCubeCoord( lat_deg, lon_deg);
}
			  
void VO_Point::LoadBinary( SDL_RWops *r_ops)
{
   float lon_lat_deg[2];
	
   if ( SDL_RWread( r_ops, lon_lat_deg, sizeof(lon_lat_deg), 1) != 1 )
     {
	throw string( "VO_Point: (LoadBinary) can't read coordinates");
     }
   
   lon_deg = lon_lat_deg[0];
   lat_deg = lon_lat_deg[1];  

   coord = PolarToCoord3f( lat_deg, lon_deg, RADIUS);
   cube_coord = PolarToCubeCoord( lat_deg, lon_deg);
}

void VO_Point::SaveASCII( ofstream *w_stream)
{
   try
     {
	*w_stream << lon_deg;
	*w_stream << lat_deg;
     }
   catch(...)
     {
	throw string( "VO_Point: (SaveASCII) can't write coordinates");
     }   
}

void VO_Point::SaveBinary( SDL_RWops *w_ops)
{
   float lon_lat_deg[2];
   
   lon_lat_deg[0] = lon_deg;
   lon_lat_deg[1] = lat_deg ;
   
   try
     {
	SDL_RWwrite( w_ops, lon_lat_deg, sizeof(lon_lat_deg), 1) == 1;
     }
   catch(...)
     {
	throw string( "VO_Point: (SaveBinary) can't write coordinates");
     }
}

void VO_Point::glDraw()
{
   glBegin( GL_POINTS);
   
   glVertex3fv( coord.GetCoords());

   glEnd();
}


/* === VO_PolyLine implemetation ===================================================== */

VO_PolyLine::~VO_PolyLine()
{
}

void VO_PolyLine::AddPoint( VO_Point point)
{
   points.push_back( point);
}

void VO_PolyLine::LoadASCII( ifstream *r_stream)
{
   while( ! r_stream->eof() )
     {
	VO_Point p;
	
	p.LoadASCII( r_stream);
	AddPoint( p);	
     }
}

void VO_PolyLine::LoadBinary( SDL_RWops *r_ops)
{	
   unsigned int nb_points = SDL_ReadLE32( r_ops);
	   
   for( unsigned int point_no=0 ; point_no < nb_points ; point_no++) 
     {	
	VO_Point p;
	
	p.LoadBinary( r_ops);
	AddPoint( p);
     }
}

void VO_PolyLine::SaveASCII( ofstream *w_stream)
{
   try
     {
	*w_stream << points.size() << endl;
     }
   catch(...)
     {
	throw string( "VO_PolyLine: (SaveASCII) can't write line length");
     }
   
   for ( unsigned int point_no=0; point_no < points.size() ; point_no++)
     {
        points[point_no].SaveASCII( w_stream);
     }
}
			     
void VO_PolyLine::SaveBinary( SDL_RWops *w_ops)
{

   SDL_WriteLE32( w_ops, points.size());
   
   for ( unsigned int point_no=0; point_no < points.size() ; point_no++)
     {
        points[point_no].SaveBinary( w_ops);
     }   
}

  
void VO_PolyLine::glDraw()
{
   glBegin( GL_LINE_STRIP);
   
   for( unsigned int p=0 ; p<points.size(); p++)
     glVertex3fv( points[p].coord.GetCoords());
   
   glEnd();  
}

inline float cube_coord_dist( Cube::Coord p1, Cube::Coord p2)
{
   return sqrtf( (p1.c[0]-p2.c[0])*(p1.c[0]-p2.c[0]) + 
		 (p1.c[1]-p2.c[1])*(p1.c[1]-p2.c[1])  );
}


VO_PolyLine *VO_PolyLine::Resample( float precision)
{
   VO_PolyLine *resampled_line = new VO_PolyLine();
   
   if ( points.size() < 2 )
     return resampled_line;
   
   resampled_line->AddPoint( points[0]);
   
   VO_Point *last_point = &points[0];
   
   for( unsigned int p=1 ; p < points.size() ; p++)
     {
   
	if ( cube_coord_dist( last_point->cube_coord, points[p].cube_coord) > precision)
	  {
	     resampled_line->points.push_back( points[p]);
	     last_point = &points[p];
	  }	

     }

   //fprintf( stderr, "VO_PolyLine: resampled from %d to %d points\n", points.size(), resampled_line->points.size()); 
   
   return resampled_line;
}

/* === VO_PolyPolyLine implementation ================================================ */

VO_PolyPolyLine::VO_PolyPolyLine()
{
}

VO_PolyPolyLine::~VO_PolyPolyLine()
{
   for( unsigned int line_no=0 ; line_no < lines.size() ; line_no++)
     {
	delete lines[line_no];
     }   
}

void VO_PolyPolyLine::AddPolyLine( VO_PolyLine *polyline)
{
   lines.push_back( polyline);
}

bool VO_PolyPolyLine::LoadFromTxtFile( string filename)
{
   FILE *f;
   char line[1024];
   float lat_deg, lon_deg;
   VO_PolyLine *current_line;   
   
   f = fopen( filename.c_str(), "r");
   
   if ( f == NULL )
     return false;
   
   while( !feof( f))
     {
	fgets( line, 1024, f);
	
	if ( line[0] == '#')
	  {
	     current_line = new VO_PolyLine();
	     lines.push_back( current_line);
	  }
	else
	  {
	     
	     if ( sscanf( line, "%f %f", &lon_deg, &lat_deg) == 2 ) 
	       current_line->AddPoint( VO_Point( lat_deg, lon_deg));
	     else
	       fprintf( stderr, "VO_PolyPolyLine: parse error\n");
	  }
     }
   
   fclose( f);
   
#ifdef _DEBUG_VERBOSE
   
   fprintf( stderr, "VO_PolyPolyLine: file %s loaded\n", filename.c_str());

#endif /* _DEBUG_VERBOSE */
}

bool VO_PolyPolyLine::SaveToTxtFile( string filename)
{
   FILE *f;
   
   f = fopen( filename.c_str(), "w");
   
   if ( f == NULL )
     return false;
   
   for( unsigned int l = 0 ; l < lines.size() ; l++)
     {
	fprintf( f, "# - new line\n");
	
	for( unsigned int p = 0 ; p < lines[l]->points.size(); p++)
	  fprintf( f, "%f %f \n", lines[l]->points[p].lon_deg,
		                 lines[l]->points[p].lat_deg);
     }
   
   fclose( f);

   fprintf( stderr, "VO_PolyPolyLine: file %s saved\n", filename.c_str());
}

bool VO_PolyPolyLine::LoadFromBinFile( string filename)
{
   SDL_RWops *file;
   
   file = SDL_RWFromFile( filename.c_str(), "rb");
   
   if ( file == NULL) 
     {
	
#ifdef _DEBUG_VERBOSE
	cerr << "VO_PolyPolyLine: can't open file " << filename << endl; 
#endif /* _DEBUG_VERBOSE */
	return false;
     }
   
   LoadBinary( file);
   
   SDL_RWclose( file);
   
   return true;
}

void VO_PolyPolyLine::LoadBinary( SDL_RWops *r_ops)
{   
   unsigned int nb_lines = SDL_ReadLE32( r_ops);
   
   for( unsigned int line_no = 0 ; line_no < nb_lines ; line_no++)
     {
	
	VO_PolyLine *current_line = new VO_PolyLine();
	
	current_line->LoadBinary( r_ops);
	lines.push_back( current_line);
     }
}

bool VO_PolyPolyLine::SaveToBinFile( string filename)
{
   SDL_RWops *file;
   
   file = SDL_RWFromFile( filename.c_str(), "wb");
   
   if ( file == NULL) 
     {
	cerr << "VO_PolyPolyLine: can't open file for writting" << filename << endl; 
	return false;
     }
  
   SaveToBinRWops( file);
   
   SDL_RWclose( file);
   
   return true;
}

bool VO_PolyPolyLine::SaveToBinRWops( SDL_RWops *w_ops)
{
   SDL_WriteLE32( w_ops, lines.size());
   
   for( unsigned int line_no=0 ; line_no < lines.size() ; line_no++)
     {
	
	SDL_WriteLE32( w_ops, lines[line_no]->points.size());
	
	for( unsigned int point_no=0 ; point_no < lines[line_no]->points.size() ; point_no++) 
	  {
	     
	     float lon_lat_deg[2];
	
	     lon_lat_deg[0] = lines[line_no]->points[point_no].lon_deg;
	     lon_lat_deg[1] = lines[line_no]->points[point_no].lat_deg ;
	     
	     // TODO: Call VO_Point::SaveToRWops ???
     	     
	     SDL_RWwrite( w_ops, lon_lat_deg, sizeof(lon_lat_deg), 1);
	  
	  }
     }
   return true;
}

void VO_PolyPolyLine::glDraw()
{
   for( unsigned int l=0 ; l<lines.size(); l++ )
     {
#ifdef _DEBUG
	if ( options.debug_vecto)
	  glColor4fv( &(debugging_colors[4*(l%3)]));
#endif /* _DEBUG */
	lines[l]->glDraw();
     }
}

VO_PolyPolyLine *VO_PolyPolyLine::Resample( float precision)
{
   VO_PolyPolyLine *resampled_line_set = new VO_PolyPolyLine();
   
   for( unsigned int line_no=0 ; line_no < lines.size() ; line_no++)
     {
	VO_PolyLine *resampled_line = lines[line_no]->Resample( precision);
	
	if ( resampled_line->getNbPoints() > 1 )
	  resampled_line_set->lines.push_back( resampled_line);
	else
	  delete resampled_line;
     }
   
   return resampled_line_set;
}

void VO_PolyPolyLine::LoadASCII( ifstream *r_stream)
{
}

void VO_PolyPolyLine::SaveASCII( ofstream *w_stream)
{
}

void VO_PolyPolyLine::SaveBinary( SDL_RWops *w_ops)
{
   SDL_WriteLE32( w_ops, lines.size());
   
   for( unsigned int line_no=0 ; line_no < lines.size() ; line_no++)
     {

	lines[line_no]->SaveBinary( w_ops);
	
/*	SDL_WriteLE32( w_ops, lines[line_no]->points.size());
	
	for( unsigned int point_no=0 ; point_no < lines[line_no]->points.size() ; point_no++) 
	  {
	     
	     float lon_lat_deg[2];
	
	     lon_lat_deg[0] = lines[line_no]->points[point_no].lon_deg;
	     lon_lat_deg[1] = lines[line_no]->points[point_no].lat_deg ;
	     
	     // TODO: Call VO_Point::SaveToRWops ???
     	     
	     SDL_RWwrite( w_ops, lon_lat_deg, sizeof(lon_lat_deg), 1);
*/	  
	  }

}
