#include <SDL.h>
#include <SDL_endian.h>
#include "gl_stuff.hpp"
#include "sdl_stuff.hpp"
#include "VectObjects.hpp"
#include "VectInfos.hpp"
#include "Options.hpp"

#define MAGIC_VECTINFO_COLLECTION   ( 'V' << 24 | 'I' << 16 | 'N' << 8 | 'F' ) 
#define MAGIC_VO_POINT              0x1
#define MAGIC_VO_LINE               0x2
#define MAGIC_VO_POLYLINE           0x3
#define MAGIC_VO_POLYPOLYLINE       0x4
#define MAGIC_VP_STRING             0x1
#define MAGIC_VP_UINT32             0x2

/* ==== VectProperties implementation ===================================== */

void VectProperty::LoadBinary( SDL_RWops *r_ops)
{
   type_magic = SDL_ReadLE32( r_ops);
   
   switch( type_magic)
     {
      case MAGIC_VP_STRING:
	string_value = sdl_ReadBinaryString( r_ops);	
	break;
      case MAGIC_VP_UINT32:	
	uint32_value = SDL_ReadLE32( r_ops);	
	break;	
      default:
	throw string( "VectProperty: (LoadBinary) type ") + "type_magic" + string(" not supported");
     }
}

void VectProperty::SaveBinary( SDL_RWops *w_ops)
{
   SDL_WriteLE32( w_ops, type_magic);	
   switch( type_magic)
     {
      case MAGIC_VP_STRING:
	sdl_WriteBinaryString( w_ops, string_value);
	break;
      case MAGIC_VP_UINT32:
	SDL_WriteLE32( w_ops, uint32_value);
	break;
      default:
	throw string( "VectProperty: (SaveBinary) type ") + "type_magic" + string(" not supported");
     }
}

void VectProperties::LoadBinary( SDL_RWops *r_ops)
{
   int nb_properties = SDL_ReadLE32( r_ops);
   
   nb_properties = 0;

   for( unsigned int p = 0 ; p < nb_properties ; p++)
     {
	VectProperty *property = new VectProperty();
	property->LoadBinary( r_ops);
	properties[property->name] = property;
     }
}

void VectProperties::SaveBinary( SDL_RWops *w_ops)
{   
   SDL_WriteLE32( w_ops, (int)properties.size());

   map<string, VectProperty*>::iterator current = properties.begin();
   map<string, VectProperty*>::iterator last    = properties.end();
   
   while( current != last)
     {
	current->second->SaveBinary( w_ops);
	current++;
     }
}

void VectProperties::Add( string property_name, VectProperty *property)
{
   properties[property_name] = property;
}

void VectProperties::Remove( string property_name)
{
   cerr << "VectProperties::Remove not implemented" << endl;
}

bool VectProperties::HaveString( string property_name, string *property_value)
{
   map< string, VectProperty*>::iterator vp_iterator;
   
   vp_iterator = properties.find( property_name);
   
   if ( vp_iterator != properties.end() && 
	vp_iterator->second->type_magic == MAGIC_VP_STRING) 
     {
	*property_value = vp_iterator->second->string_value;
	*property_value = properties[property_name]->string_value;
	return true;
     }
   else  
     {
	*property_value = properties[property_name]->string_value;

	//fprintf( stderr, "not found in %d\n", properties.size());
	return false;
     }
}

/* ==== VectInfo implementation =========================================== */

VectInfo::VectInfo()
{
   vo = NULL;
   vp = NULL;
}

VectInfo::VectInfo( VectObject *vo, VectProperties *vp, string type, string name)
{
   this->type = type;
   this->name = name;
   this->vo = vo;
   this->vp = vp;
}

VectInfo::~VectInfo()
{
   if ( vo)
     delete vo;
   if ( vp)
     delete vp;
}

/*
string VectInfo::GetTypeString( unsigned int magic_vo_point)
{
}

unsigned int VectInfo::GetTypeMagic( string type)
{
}
*/

void VectInfo::CreateVectObject( string type)
{
   this->type = type;

   if ( type == string( "VO_POINT"))
     {
	vo = new VO_Point();
     }
   else if ( type == string( "VO_POLYLINE"))
     {
	vo = new VO_PolyLine();
     }
   else if ( type == string( "VO_POLYPOLYLINE"))
     {
	vo = new VO_PolyPolyLine();
     }
   else
     {
	throw string("VectItem: type \"" + type + "\" unknown or not supported !");
     }
}

void VectInfo::LoadASCII( ifstream *r_stream)
{
   try
     {
	*r_stream >> type;
	*r_stream >> name;
     }
   catch(...)
     {
	throw string( "VectInfo: (LoadASCII) can't read properties");
     }
      
   CreateVectObject( type);
   
   vo->LoadASCII( r_stream);
}

void VectInfo::LoadBinary( SDL_RWops *r_ops)
{
   try
     {
	type = sdl_ReadBinaryString( r_ops);
	name = sdl_ReadBinaryString( r_ops);
     }
   catch(...)
     {
	throw string( "VectInfo: (LoadBinary) can't read header");	
     }
   
   CreateVectObject( type);
   vo->LoadBinary( r_ops);   
   
   vp = new VectProperties();
   vp->LoadBinary( r_ops);
}


void VectInfo::SaveASCII( ofstream *w_stream)
{
   try
     {	
	*w_stream << "\"" << type << "\"" << endl;
	*w_stream  << "\"" << name << "\"" << endl;
     }
   catch(...)
     {
	throw string( "VectInfo: (SaveASCII) can't save header");
     }
   //vp->SaveASCII( w_stream);
   vo->SaveASCII( w_stream);
}

void VectInfo::SaveBinary( SDL_RWops *w_ops)
{
   try
     {
	sdl_WriteBinaryString( w_ops, type);
	sdl_WriteBinaryString( w_ops, name);	
     }
   catch(...)
     {
	throw string( "VectInfo: (SaveBinary) can't save header");
     }
   
   vo->SaveBinary( w_ops);
   
   vp->SaveBinary( w_ops);
}


void VectInfo::Render()
{
   // TODO: use property
   
   vo->glDraw();
}

/* ==== VectInfoCollection implementation ================================= */

VectInfoCollection::VectInfoCollection()
{
}

VectInfoCollection::~VectInfoCollection()
{
   for( unsigned int vi = 0 ; vi < collection.size(); vi++)
     delete collection[vi];
}

void VectInfoCollection::AddVectInfo( VectInfo *vi)
{
   collection.push_back( vi);
}

void VectInfoCollection::LoadASCII( ifstream *r_stream)
{
}

void VectInfoCollection::LoadBinary( SDL_RWops *r_ops)
{
   if ( SDL_ReadLE32( r_ops) != MAGIC_VECTINFO_COLLECTION)
     {
	throw string( "VectInfoCollection: not a VectInfoCollection file\n");
     }
   
   unsigned int nb_vi = SDL_ReadLE32( r_ops);
   
   for( unsigned int vi = 0 ; vi < nb_vi ; vi++)
     {
	VectInfo *_vi = new VectInfo();
	_vi->LoadBinary( r_ops);
	collection.push_back( _vi);
     }
}

void VectInfoCollection::SaveASCII( ofstream *w_stream)
{
}

void VectInfoCollection::SaveBinary( SDL_RWops *w_ops)
{
   SDL_WriteLE32( w_ops, MAGIC_VECTINFO_COLLECTION);
   SDL_WriteLE32( w_ops, collection.size());
   
   for( unsigned int i = 0 ; i < collection.size() ; i++)
     collection[i]->SaveBinary( w_ops);  
}

void VectInfoCollection::Render()
{
   for( unsigned int vi = 0 ; vi < collection.size(); vi ++ )
     {
	// TODO: Make this generic //
	if ( collection[vi]->GetName() == string("Boundaries") && options.draw_boundaries)
	 {
	    glLineStipple( 1, 0x00ff);
	    glEnable( GL_LINE_STIPPLE);
	    glDisable( GL_TEXTURE_2D);
	    glDisable( GL_BLEND);
	    glColor3f( BOUNDARIES_COLOR);
	    glLineWidth( 2.0f);
	    glEnable( GL_LINE_SMOOTH);	 
	    collection[vi]->Render();
	    glDisable( GL_LINE_STIPPLE);
	 }  
	else if ( collection[vi]->GetName() == string("Coastlines") && options.draw_coastlines)
	  {
	     glDisable( GL_TEXTURE_2D);
	     glDisable( GL_BLEND);
	     glColor3f( COASTLINES_COLOR);
	     glLineWidth( 1.0f);
	     glEnable( GL_LINE_SMOOTH);
	     
	     collection[vi]->Render();
	  }
	else if ( collection[vi]->GetName() == string("Rivers") &&  options.draw_rivers)
	  {
	     glDisable( GL_TEXTURE_2D);
	     glDisable( GL_BLEND);
	     glColor3f( RIVERS_COLOR);
	     glLineWidth( 1.0f);
	     glEnable( GL_LINE_SMOOTH);
	     
	     collection[vi]->Render();
	  }
     }
}
