#ifndef _TEXTURE_HPP
#define _TEXTURE_HPP

/** \file Texture.hpp
 * An OpenGL Texture Object implementation.
 * Loading the data from disk and uploading the data in video memory tasks 
 * have been splited in two separeted functions in order to make it possible
 * to load asynchronously the texture from disk (i.e. loading can be done in
 * a one thread without slowing the OpenGL one).
 * 
 * This implementation rely on SDL_image library so that it can read RGB
 * textures in all formats supported by SDL_image library (i.e. BMP, PNG,
 * JPG, TGZ, PCX...) and OpenGL compressed textures. 
 **/

#include <string>
#include <SDL.h>
#include "gl_stuff.hpp"
#include "config.h"

using namespace std;

struct CompressedLod;

class Texture
{
 public:
   Texture();
   ~Texture();
  
   // Loading from File methods
   
   void LoadImage( string filename);
   void LoadCompressed( string filename);
   void LoadRaw( string filename, unsigned short width, unsigned short height);

   // Loading from SDL_RW_ops methods

   void LoadImage( SDL_RWops *rwops);
   void LoadCompressed( SDL_RWops *rwops); 
   void LoadRaw( SDL_RWops *rwops, unsigned int width, unsigned int height);
   
   // Saving to File methods
   
   void SaveBMP( string filename);
   void SaveCompressed( string filename);
   void SaveRaw( string filename);
   inline void Save( string filename) { SaveBMP( filename);}
   
   // Saving to SDL_RW_ops methods
 
   void SaveImage( SDL_RWops *rwops);
   void SaveCompressed( SDL_RWops *rwops);
   void SaveBMP( SDL_RWops *rwops);
   
   // Operations
   
   void CreateFromScratch( unsigned short width, unsigned short height, GLenum format);
   void Resample( unsigned short new_width, unsigned short new_width);
   void Bind();
   void UpLoad();
   void UpLoad( GLuint texture_object);
   
   // in system memmory texture buffer accessors
   
   enum Composant{ R=0, G=1, B=2};
   
   inline GLuint GetTextureObject() const { return texture_object;}
	
   inline unsigned char GetPixelComp( unsigned short x, unsigned short y, unsigned short comp) 
     { return ((unsigned char*)image->pixels)[y*image->pitch+x*image->format->BytesPerPixel+comp];}
   
   inline void SetPixelComp( unsigned short x, unsigned short y, Composant comp, unsigned char value) 
     { ((unsigned char*)image->pixels)[y*image->pitch+x*image->format->BytesPerPixel+comp] = value;}
   
   inline unsigned char *GetPixelPtr( unsigned short x, unsigned short y) 
     { return ((unsigned char*)image->pixels)+y*image->pitch+x*image->format->BytesPerPixel;}
   
   inline void *GetPixelsPtr() { return image->pixels;}
	
   inline bool IsReady() { return preloaded;}
   
 private:
   
   GLuint size;
   GLuint texture_object;
   bool compressed;
   bool preloaded;
   bool uploaded;
   bool uploaded_sub; // true when glTexSubImage2D is called on a persistent texture
   unsigned char *data;
   SDL_Surface *image;   
   CompressedLod *compressed_lod;

   // opengl texture paramters ( if not a compressed texture)
   GLsizei width;
   GLsizei height;
   GLuint border;
   GLint internal_format;
   GLenum format;
   GLenum type;
   GLint nb_lod; // TODO: keep this?
};

#endif
