#include <iostream>
#include <string.h>
#include <SDL.h>
#include <SDL_endian.h>
#include <SDL_image.h>
#include "Texture.hpp"
#include "gl_stuff.hpp"
//#include "sdlzlib.h"

struct CompressedLod
{
   GLuint width;
   GLuint height;
   GLuint border;
   GLuint size;
   GLuint internal_format; // 0 is uncompressed
   unsigned char *data;
};

Texture::Texture()
{
   preloaded = false; 
   uploaded = false;
   uploaded_sub = false;
   texture_object = 0;
   data = NULL;
   image = NULL;
   compressed = false;
}

Texture::~Texture()
{
   if ( !compressed && preloaded)
     {
	SDL_FreeSurface( image) ;
     }
   
   if ( preloaded)
     {
	if ( !compressed)
	  {
	     delete[] data;
	  }
	else
	  {
	     for(int lod=0;lod<nb_lod;lod++)
	       {
		  delete[] compressed_lod[lod].data;
	       }
	     delete[] compressed_lod;	
	  }
     }
   
   if ( uploaded && !uploaded_sub)
     {
	gl_plan_destroying( OBJECT_GL_TEXTURE_2D, texture_object);
     }
}


void Texture::LoadImage( SDL_RWops *rwops)
{
   
   image = IMG_Load_RW( rwops, 0);
   
   if ( image == NULL ) 
     {
	cerr << "Texture: unable to load texture ( " << SDL_GetError() << " )" << endl;
	return;
     }

   if ( image->format->BytesPerPixel == 3 &&
	image->format->Bmask == 0xFF0000  &&
	image->format->Gmask == 0x00FF00  &&
	image->format->Rmask == 0x0000FF  &&
	image->format->Amask == 0x000000     )
       {
	  internal_format = GL_RGB;
	  format = GL_RGB;
	  type = GL_UNSIGNED_BYTE;
       }
   else if ( image->format->BytesPerPixel == 3 &&
	     image->format->Rmask == 0xFF0000  &&
	     image->format->Gmask == 0x00FF00  &&
	     image->format->Bmask == 0x0000FF  &&
	     image->format->Amask == 0x000000    
	     )
     {
	internal_format = GL_RGB;
	format = GL_BGR;
	type = GL_UNSIGNED_BYTE;
     }
   else if ( image->format->BytesPerPixel == 1 )
     {
	internal_format = GL_LUMINANCE;
	format = GL_LUMINANCE;
	type = GL_UNSIGNED_BYTE;
     }
   else
     {
	cerr << "Texture: format not supported so convert in RGB 24 bits" << endl;
	cerr << "WARNING: This may slow down your application (to avoid this, store texture in a supported format)" << endl;
	
	SDL_Surface *image_tmp = image;
	SDL_PixelFormat glPixelFormat;

	glPixelFormat.palette = NULL;
	glPixelFormat.BitsPerPixel = 24;
	glPixelFormat.BytesPerPixel = 3;
	glPixelFormat.Bmask = 0xFF0000;
	glPixelFormat.Gmask = 0x00FF00;
	glPixelFormat.Rmask = 0x0000FF;
	glPixelFormat.Amask = 0x000000;
	glPixelFormat.Bshift = 0x10; 
	glPixelFormat.Gshift = 0x8; 
	glPixelFormat.Rshift = 0x0; 
	glPixelFormat.Ashift = 0;
	glPixelFormat.Rloss = 0; 
	glPixelFormat.Gloss = 0; 
	glPixelFormat.Bloss = 0; 
	glPixelFormat.Aloss = 0x8;
	glPixelFormat.colorkey = 0x0;
	glPixelFormat.alpha = 0xFF;
	
	image = SDL_ConvertSurface( image_tmp, &glPixelFormat, SDL_SWSURFACE);
	
	if ( image == NULL ) 
	  {
	     cerr << "Texture: conversion failed" << endl;
	     SDL_FreeSurface( image_tmp);
	     return;
	  }
	
	SDL_FreeSurface( image_tmp);

	internal_format = GL_RGB;
	format = GL_RGB;
	type = GL_UNSIGNED_BYTE;
     }
		
   width = image->w;
   height = image->h;
   border = 0;
   preloaded = true;
   uploaded = false;
   compressed = false;
}

void Texture::LoadImage( string filename)
{
   SDL_RWops *file = SDL_RWFromFile( filename.c_str(), "rb");

   //SDLZlib_RWFromFile   
   
   if ( file == NULL )
     {
	cerr << "Texture: load of " << filename << " failed" << endl;	
	return;
     }
   
   LoadImage( file);
   
   SDL_RWclose( file);
}

void Texture::SaveBMP( SDL_RWops *rwops)
{
}

void Texture::SaveBMP( string filename)
{
   SDL_SaveBMP( image, filename.c_str()) ;
}

void Texture::SaveCompressed( SDL_RWops *rwops)
{
}

#ifdef USE_TEXTURE_COMPRESSION

void Texture::LoadCompressed( SDL_RWops *f)
{
   if ( f == NULL)
     return; 
     
   char signature[2];
  
   SDL_RWread( f, signature, sizeof(char), 2);
   
   if ( strncmp( signature, "CT", 2) != 0 )
     {
	cerr << "Texture::LoadCompressed: not a compressed texture" << endl; 
	SDL_RWclose( f);
	return;
     }
   
   nb_lod = SDL_ReadLE16( f);
   
   compressed_lod = new CompressedLod[nb_lod];
   
   for( GLint current_lod = 0; current_lod < nb_lod ; current_lod++)
     {   
	CompressedLod *cl = &(compressed_lod[current_lod]);

	cl->width = SDL_ReadLE16( f);
       	cl->height = SDL_ReadLE16( f);
	cl->internal_format = SDL_ReadLE32( f);
	cl->size = SDL_ReadLE32( f);
	cl->border = 0;
	cl->data = new unsigned char[cl->size];	
	SDL_RWread( f, cl->data, sizeof(unsigned char), cl->size); 
     }
   
   preloaded = true;
   compressed = true;
}

void Texture::LoadCompressed( string filename)
{
   SDL_RWops *file;
  
   file = SDL_RWFromFile( filename.c_str(), "rb");
   
   if ( file == NULL) 
     {
	cerr << "Texture: can't open file " << filename << endl; 
	return;
     }
   
   LoadCompressed( file);
   
   SDL_RWclose( file);
}

#endif

void Texture::CreateFromScratch( unsigned short width, unsigned short height, GLuint format)
{
   image = SDL_CreateRGBSurface( SDL_SWSURFACE, 
				 width, 
				 height,
				 24,
				 0x00FF0000, 
				 0x0000FF00,
				 0x000000FF,
				 0);   
   
   if ( image == NULL ) 
     {
	cerr << "Texture::CreateFromScratch: allocation failed!" << endl;
	return;
     }
   
   this->width = image->w;
   this->height = image->h;
   border = 0;
   internal_format = GL_RGB;
   format = GL_RGB;
   type = GL_UNSIGNED_BYTE;
   preloaded = true;
   compressed = false;
}

void Texture::Resample( unsigned short new_width, unsigned short new_height)
{
   SDL_Rect src_rect, dst_rect;
   SDL_Surface *new_image;
   
   new_image = SDL_CreateRGBSurface( SDL_SWSURFACE, 
				 new_width, 
				 new_height,
				 24,
				 0x00FF0000, 
				 0x0000FF00,
				 0x000000FF,
				 0);   
   
   if ( new_image == NULL ) 
     {
	cerr << "Texture::Resample: allocation failed!" << endl;
	return;
     }
   
   src_rect.x = 0;
   src_rect.y = 0;
   src_rect.w = image->w;
   src_rect.h = image->h;
   
   dst_rect.x = 0;
   dst_rect.y = 0;
   dst_rect.w = new_image->w;
   dst_rect.h = new_image->h;
   
   SDL_SoftStretch( image, &src_rect, new_image, &dst_rect);
                    
   SDL_FreeSurface( image);

   image = new_image;
   this->width = image->w;
   this->height = image->h;
}

void Texture::UpLoad()
{   
   if ( !preloaded || uploaded)
     return;
   
   glGenTextures( 1, &texture_object) ;
   
   glBindTexture( GL_TEXTURE_2D, texture_object) ;
   
   glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
   
   glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
   glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
   
   if ( !compressed)
     {
	
	glTexParameteri( GL_TEXTURE_2D, GL_GENERATE_MIPMAP_SGIS, GL_TRUE);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,   GL_LINEAR);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,   GL_LINEAR_MIPMAP_LINEAR);
	glPixelStorei( GL_UNPACK_ALIGNMENT, 1);
	
	glTexImage2D( GL_TEXTURE_2D, 0, 3, image->w, image->h, 0, format, type, image->pixels);
	
     }
   else
     {
	
#ifdef USE_TEXTURE_COMPRESSION 

	glTexParameteri( GL_TEXTURE_2D, GL_GENERATE_MIPMAP_SGIS, GL_FALSE); // seem's to be very slow
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,   GL_LINEAR);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,   (nb_lod>1)?GL_LINEAR_MIPMAP_LINEAR:GL_LINEAR);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_BASE_LEVEL,   0);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAX_LEVEL,    4);

	// as GL_GENERATE_MIPMAP_SGIS is slow with compressed texture 
	// we have to upload every lod.
	
	for( GLint current_lod=0 ; current_lod<nb_lod ; current_lod++)
	  {	     
	     CompressedLod *cl = &(compressed_lod[current_lod]);
	     
	     glCompressedTexImage2D( GL_TEXTURE_2D, current_lod, cl->internal_format, 
				     cl->width, cl->height, 0, cl->size, cl->data);
	  }
	
#endif /* USE_TEXTURE_COMPRESSION */
	
     }
   
   if ( glGetError() == 0) 
     uploaded = true;
   else
     throw string( "Texture: can't upload Texture into video memory");
}

void Texture::UpLoad( GLuint texture_object)
{   
   if ( !preloaded || uploaded)
     return;
      
   this->texture_object = texture_object;
   
   glBindTexture( GL_TEXTURE_2D, texture_object) ;
   
   if ( !compressed)
     {	
	glTexSubImage2D( GL_TEXTURE_2D, 0, 0, 0, image->w, image->h, format, type, image->pixels);		  
     }
   else
     {
	
#ifdef USE_TEXTURE_COMPRESSION 
	
	for( GLint current_lod = 0; current_lod<nb_lod; current_lod++)
	  {	     
	     CompressedLod *cl = &(compressed_lod[current_lod]);
	     
	     glCompressedTexSubImage2D( GL_TEXTURE_2D, current_lod, 0, 0, cl->width, cl->height, 0, cl->size, cl->data);
	  }
	
#endif /* USE_TEXTURE_COMPRESSION */	
	
     }
   
   if ( glGetError() == 0) 
     uploaded_sub = uploaded = true;
   else
     throw string( "Texture: can't upload texture into video memory");
}

void Texture::Bind()
{
   if ( !preloaded)
     return;
   
   if ( !uploaded)
     UpLoad();
   
   if ( glIsTexture( texture_object))
     glBindTexture( GL_TEXTURE_2D, texture_object);
   else
     cerr << "Texture::Bind: sorry " << texture_object << " is not a texture object" << endl;
}
