#include <stdlib.h>
#include <stdio.h>
#include <string>
#include <math.h>
#include <iostream>
#include <SDL.h>
#include "math/Vector.hpp"
#include "gl_stuff.hpp"
#include <GL/glu.h>
#include "sdl_stuff.hpp"
#include "Planet.hpp"
#include "projection.hpp"
#include "Cache.hpp"
#include "Texture.hpp"
#include "Chunk.hpp"
#include "Grid.hpp"
#include "VirtualChunk.hpp"
#include "Camera.hpp"
#include "Options.hpp"
#include "config.h"
#include "VectObjects.hpp"
#include "BufferObject.hpp"

#define EPS 0.00001
#define SEA_COLOR 0.023529412f,0.031372549f,0.188235294

Planet::Planet()
{	
   cache = NULL;
   grids = NULL;
   normals_grids = NULL;
}

Planet::~Planet()
{
   if ( cache)
     delete cache;
   if ( grids)
     delete[] grids;
   if ( normals_grids)
     delete[] normals_grids;
}

void Planet::Initialize( string filename)
{
   cache = new Cache( CACHE_SIZE);
   cache->Initialize( this);

   ds.Open( filename);
  
   LoadSetup();
     
   grid_width = ( 1 << (nb_lod+1) );
   
#ifdef USE_PRECOMPUTED_GRID
   
   ComputeGrids();

#endif /* USE_PPECOMPUTED_GRID */

}

void Planet::LoadSetup()
{
   // TODO: Load properties with the help of a parser ( xml ? )
    
   nb_lod = 5;
   textures_width = 500;
   radius = 0.5f;
   dataset_version = 0;
   
   SDL_RWops *cfg_file;
   
   if ( ds.isReady())   
     cfg_file = ds.OpenFile( "config/planet.cfg");
   else
     cfg_file = SDL_RWFromFile( "dataset/config/planet.cfg", "rb");
   
   if ( cfg_file == NULL )
     throw string( "Planet: Configuration file not present in dataset");
   
   char line[1024];
   int line_no = 0;
   
   while ( SDL_RW_gets( line, 1024, cfg_file))
     {
	line_no += 1;
	char parameter_name[1024];
	char parameter_value[1024];
	
	if ( sscanf( line, "%s = \"%[^\"]\"", parameter_name, parameter_value ) == 2 )
	  {
	     if ( string( parameter_name) == string("title"))
	       {
		  ;
	       }
	     else if ( string( parameter_name) == string("dataset_version"))
	       {
		  dataset_version = atoi( parameter_value);
	       }
	     else if ( string( parameter_name) == string("nb_lod"))
	       {
		  nb_lod = atoi( parameter_value);
	       }
	     else if ( string( parameter_name) == string("textures_width"))
	       {
		  textures_width = atoi( parameter_value);
	       }
	     else 
	       {
		  cerr << "Planet: line " << line_no << " : unknown parameter \"" << parameter_name << "\"" << endl;
		  cerr << "        Perhaps you should upgrade TinyPlanet in order to be able to access" << endl;
		  cerr << "        all data provided by this dataset" << endl;
	       }
	  }
	
      }
   
   SDL_RWclose( cfg_file);
   
#ifndef DEBUG_VERBOSE
   
   cout << "Planet: Configuration loaded successfully" << endl;
   cout << "        Number of levels: " << nb_lod << endl;
   cout << "        Textures size:    " << textures_width << "x" << textures_width << endl;
   cout << "        Dataset version:  " << dataset_version <<  "( expected version " << DATASET_VERSION << " )" << endl;

#endif /* DEBUF_VERBOSE */

   if ( dataset_version < DATASET_VERSION)
     {
	cerr << "Planet: WARNING: dataset format is out of date !" << endl;
	cerr << "        updating the dataset is recomanded." << endl;
     }
   else if( dataset_version > DATASET_VERSION)
     {
	cerr << "Planet: WARNING: dataset format is " << dataset_version << " while " << DATASET_VERSION << " was expected!" << endl;
	cerr << "        You may miss some features from this dataset" << endl;
	cerr << "        updating the program is recomanded." << endl;
     }
}

void Planet::ComputeGrids()
{
   // Precompute grids
       
   cout << "Planet: Precompute grids of " << grid_width << "x" << grid_width << "... ";
   cout.flush();
   
   grids = new Grid< Vector3f>[6];
   normals_grids = new Grid< Vector3f>[6];

   for( unsigned int g=0;g<6;g++)
     {
	grids[g].Init( grid_width+1, grid_width+1); //= Grid<Vector3f>( grid_width+1, grid_width+1);	
	normals_grids[g].Init(grid_width+1, grid_width+1);// = Grid<Vector3f>( grid_width+1, grid_width+1);
     }
   
   // 0 Front, 1 East, 2 Back, 3 West, 4 North, 5 South  
  
   for( unsigned int gy = 0 ; gy < grid_width+1 ; gy++)
     for( unsigned int gx = 0 ; gx < grid_width+1 ; gx++)
       {
	  float x = ((float)gx) * 2.0f * CUBE_RAY / (float) grid_width;
	  float y = ((float)gy) * 2.0f * CUBE_RAY / (float) grid_width;
	
	  grids[0].Set( gx, gy, CubeToSphere3f( Vector3f( -CUBE_RAY+x,  CUBE_RAY-y,  CUBE_RAY  ))); // Front
	  grids[1].Set( gx, gy, CubeToSphere3f( Vector3f(  CUBE_RAY  ,  CUBE_RAY-y,  CUBE_RAY-x))); // East
	  grids[2].Set( gx, gy, CubeToSphere3f( Vector3f(  CUBE_RAY-x,  CUBE_RAY-y, -CUBE_RAY  ))); // Back	  
	  grids[3].Set( gx, gy, CubeToSphere3f( Vector3f( -CUBE_RAY  ,  CUBE_RAY-y,  -CUBE_RAY+x))); // West
	  grids[4].Set( gx, gy, CubeToSphere3f( Vector3f( -CUBE_RAY+x,  CUBE_RAY  , -CUBE_RAY+y))); // North
	  grids[5].Set( gx, gy, CubeToSphere3f( Vector3f(  CUBE_RAY-x, -CUBE_RAY  , CUBE_RAY-y))); // South
	  
       	  normals_grids[0].Set( gx, gy, grids[0].Get( gx, gy));
	  normals_grids[1].Set( gx, gy, grids[1].Get( gx, gy));
	  normals_grids[2].Set( gx, gy, grids[2].Get( gx, gy));
	  normals_grids[3].Set( gx, gy, grids[3].Get( gx, gy));
	  normals_grids[4].Set( gx, gy, grids[4].Get( gx, gy));
	  normals_grids[5].Set( gx, gy, grids[5].Get( gx, gy));

	  normals_grids[0].Get( gx, gy).normalize();
	  normals_grids[1].Get( gx, gy).normalize();
	  normals_grids[2].Get( gx, gy).normalize();
	  normals_grids[3].Get( gx, gy).normalize();
	  normals_grids[4].Get( gx, gy).normalize();
	  normals_grids[5].Get( gx, gy).normalize();	  
       }    
   
   cout << "Done" << endl;
}

float Planet::GetDistance( VirtualChunk *vchunk, Camera* camera)
{
   
#ifdef USE_PRECOMPUTED_GRID
   
   int grid_number = vchunk->node.GetFace();
   
   Vector3f center = grids[grid_number].Get( (vchunk->min_gx+vchunk->max_gx)/2, (vchunk->min_gy+vchunk->max_gy)/2);
   
#else /* USE_PRECOMPUTED_GRID not defined */
   
   Vector3f center = CubeToSphere3f( Vector3f( (vchunk->xmin+vchunk->xmax)/2.0f ,
					       (vchunk->ymin+vchunk->ymax)/2.0f ,
					       (vchunk->zmin+vchunk->zmax)/2.0f ));
  
#endif /* USE_PRECOMPUTED_GRID not defined */
   
   return sqrtf(   (center[0]-camera->GetEyePosition()[0])*(center[0]-camera->GetEyePosition()[0]) 
		+ (center[1]-camera->GetEyePosition()[1])*(center[1]-camera->GetEyePosition()[1])
		+ (center[2]-camera->GetEyePosition()[2])*(center[2]-camera->GetEyePosition()[2]));
}

float Planet::GetSurfaceProjectedScreenSize( VirtualChunk *vchunk)
{
   double modelMatrix[16];
   double projMatrix[16];
   GLint viewPort[4];
   Vector3f w1, w2, w3, w4;
   double p1x, p1y, p1z;
   double p2x, p2y, p2z;
   double p3x, p3y, p3z;
   double p4x, p4y, p4z;
   double dx, dy;
   
   // Get all transformation matrix to reverse projection
  
   glGetDoublev( GL_MODELVIEW_MATRIX, modelMatrix ) ;
   glGetDoublev( GL_PROJECTION_MATRIX, projMatrix ) ;
   glGetIntegerv( GL_VIEWPORT, viewPort) ;
   
   int grid_number = vchunk->node.GetFace(); 
   
   w1 = grids[grid_number].Get( vchunk->min_gx, vchunk->min_gy);
   w2 = grids[grid_number].Get( vchunk->max_gx, vchunk->min_gy);
   w3 = grids[grid_number].Get( vchunk->max_gx, vchunk->max_gy);
   w4 = grids[grid_number].Get( vchunk->min_gx, vchunk->max_gy);
   
   gluProject( w1[0], w1[1], w1[2], modelMatrix, projMatrix, viewPort, &p1x, &p1y, &p1z);
   gluProject( w2[0], w2[1], w2[2], modelMatrix, projMatrix, viewPort, &p2x, &p2y, &p2z);
   gluProject( w3[0], w3[1], w3[2], modelMatrix, projMatrix, viewPort, &p3x, &p3y, &p3z);
   gluProject( w4[0], w4[1], w4[2], modelMatrix, projMatrix, viewPort, &p4x, &p4y, &p4z);

   dx = fabs( p1x - p2x);
   dy = fabs( p1y - p4y);
   
   return (dx>dy)?dx:dy;
}


float Planet::ComputeVirtualChunkScreenSize( VirtualChunk *vchunk, Camera *camera)
{
   
   // Compute bounding sphere ray of VirtualChunk
    
   float bs_ray = RADIUS / (float) ( 1 << vchunk->node.GetDepth());
   
   // Compute screen space of VirtualChunk  
   
   float d = GetDistance( vchunk, camera);
   float screen_space_pix = ( bs_ray * options.screen_width ) / ( 2.0f * d * tan ( 60.0f / 180.0f * M_PI / 2.0f ));  

   return screen_space_pix;
}

void Planet::DrawSurface( VirtualChunk *vchunk, Camera *camera)
{
   float screen_space_pix = ComputeVirtualChunkScreenSize( vchunk, camera);
   
   float normalized_screen_space = ( screen_space_pix < (textures_width/2) ) ? 0.0f : ( ( screen_space_pix > textures_width ) ?  1.0f : (screen_space_pix-((float)(textures_width/2))) / ((float)(textures_width/2)));

   /*float normalized_screen_space = 0.0f;
   
   if ( screen_space_pic > ((float)textures_width/2))
     {
	if ( screen_space_pix > (float)textures_width )
	  normalized_screen_space = 1.0f;
	else
	  normalized_screen_space = ((float)(textures_width-textures_width/2))/()
     }
   */
   float bs_ray = RADIUS / (float)( 1 << vchunk->node.GetDepth());
   int grid_number = vchunk->node.GetFace();
   Vector3f center = grids[grid_number].Get( (vchunk->min_gx+vchunk->max_gx)/2, (vchunk->min_gy+vchunk->max_gy)/2);


   bool chunk_visible = camera->GetFrustum()->sphereInFrustum( center[0], center[1], center[2], bs_ray);
   
   if ( chunk_visible == false )
     return;
   
   // if chunk projected surface size is not tolerable ( bigger than texture size) 
   //   render children
   // else
   //   render this chunk
   
   if ( ( vchunk->node.GetDepth() < nb_lod ) && normalized_screen_space >= 1.0f  && chunk_visible)
     {
	VirtualChunk child;
	
	// 1st child
	child = vchunk->GetChild1();
	DrawSurface( &child, camera);
	
	// 2nd child
	child = vchunk->GetChild2();
	DrawSurface( &child, camera);
	
	// 3rd child
	child = vchunk->GetChild3();
	DrawSurface( &child, camera);

	// 4th child
	child = vchunk->GetChild4();
	DrawSurface( &child, camera);
     }
   else
     {
	DrawChunk( vchunk, normalized_screen_space);	
	
	drawn_nodes_counter++;
     }  
   
   explored_nodes_counter++;
}

//////////////////////////           ///////////////////////////////////
////////////////////////// RENDERING ///////////////////////////////////
//////////////////////////           ///////////////////////////////////

void Planet::FillChunkGeometry( Chunk *chunk)
{
   // retrieve VirtualChunk from Node
   VirtualChunk *vchunk = new VirtualChunk();
   *vchunk = GetVirtualChunk( chunk->node); 
   
   unsigned int width = vchunk->max_gx - vchunk->min_gx;
   unsigned int bo_index = 0; 
   
   // Fill BufferObject
   
   chunk->grid_width = width;
   chunk->bo = new BufferObject( (width+1)*(width+1), 3, 3, 2, 0);

   for( unsigned int gy = vchunk->min_gy ; gy < vchunk->max_gy+1 ; gy++)
     for( unsigned int gx = vchunk->min_gx ; gx < vchunk->max_gx+1 ; gx++)
       {
	  Vector3f vertex;
	  Vector3f normal;
	  // Coord2D texture_coord;
	  float texture_coord_x, texture_coord_y;
	  
	  vertex = grids[vchunk->node.GetFace()].Get( gx, gy);
	  normal = normals_grids[vchunk->node.GetFace()].Get( gx, gy);	  
	  texture_coord_x = (float) (gx - vchunk->min_gx) / (float)width;
	  texture_coord_y = (float) (gy - vchunk->min_gy) / (float)width;
	  
	  chunk->bo->SetVertex( bo_index, 0, vertex[0]);
	  chunk->bo->SetVertex( bo_index, 1, vertex[1]);
	  chunk->bo->SetVertex( bo_index, 2, vertex[2]);
	  
	  chunk->bo->SetNormal( bo_index, 0, normal[0]);
	  chunk->bo->SetNormal( bo_index, 1, normal[1]);
	  chunk->bo->SetNormal( bo_index, 2, normal[2]);
	  
	  chunk->bo->SetTexCoord( bo_index, 0, texture_coord_x);
	  chunk->bo->SetTexCoord( bo_index, 1, texture_coord_y);
	  
	  bo_index += 1;
       }   
   
   // Fill indices
   
   chunk->nb_elements = width * width * 3 * 2; // 3 vertices * 2 triangles
   chunk->indices = new GLshort[chunk->nb_elements];
   
   for( unsigned int gy=0 ; gy < width ; gy++)
     for( unsigned int gx=0 ; gx < width ; gx++)
       {
	  chunk->indices[(gy*width+gx)*6+ 0] = (gy+0)*(width+1)+(gx+0); 
	  chunk->indices[(gy*width+gx)*6+ 1] = (gy+0)*(width+1)+(gx+1); 
	  chunk->indices[(gy*width+gx)*6+ 2] = (gy+1)*(width+1)+(gx+0); 
	  chunk->indices[(gy*width+gx)*6+ 3] = (gy+0)*(width+1)+(gx+1); 
	  chunk->indices[(gy*width+gx)*6+ 4] = (gy+1)*(width+1)+(gx+1); 
	  chunk->indices[(gy*width+gx)*6+ 5] = (gy+1)*(width+1)+(gx+0); 
       }
}

void Planet::DrawChunkOcean( VirtualChunk *vchunk)
{
   glDisable( GL_TEXTURE_2D);
   glDisable( GL_BLEND);
   glColor3f( SEA_COLOR);
   glBegin( GL_QUADS);
   
#ifdef USE_PRECOMPUTED_GRID
      
   for( unsigned int gy = vchunk->min_gy ; gy < vchunk->max_gy ; gy++)
     for( unsigned int gx = vchunk->min_gx ; gx < vchunk->max_gx ; gx++)
       {
	  
	  Vector3f c1, c2, c3, c4;
	  Vector3f n1, n2, n3, n4;
	  int gn = vchunk->node.GetFace();
	  
	  c1 = grids[gn].Get( gx,   gy  );
	  c2 = grids[gn].Get( gx+1, gy  );
	  c3 = grids[gn].Get( gx+1, gy+1);
	  c4 = grids[gn].Get( gx,   gy+1);
	  
	  n1 = normals_grids[gn].Get( gx,   gy  );
	  n2 = normals_grids[gn].Get( gx+1, gy  );
	  n3 = normals_grids[gn].Get( gx+1, gy+1);
	  n4 = normals_grids[gn].Get( gx,   gy+1);
	  	  
#else /* USE_PRECOMPUTED_GRID not defined */
/*	  
  float width = ( vchunk->xmin == vchunk->xmax) ? (vchunk->ymax-vchunk->ymin) : (vchunk->xmax-vchunk->xmin);  
	    
  for( float b=0;b<width-EPS;b+=GRID_STEP)
    for( float a=0;a<width-EPS;a+=GRID_STEP)
      {
	      
	 Coord3D c1, c2, c3, c4;
	 float tax, tay, tbx, tby;
	 
	 if ( vchunk->node.GetPrefix() == 'W' || vchunk->node.GetPrefix() == 'E' )
	   {	    
	      c1 = CubeToSphere(Coord3D( vchunk->xmin,             vchunk->ymin+b,           vchunk->zmin+a     ));
	      c2 = CubeToSphere(Coord3D( vchunk->xmin,             vchunk->ymin+b          , vchunk->zmin+a+GRID_STEP    ));
	      c3 = CubeToSphere(Coord3D( vchunk->xmin,             vchunk->ymin+b+GRID_STEP, vchunk->zmin+a+GRID_STEP));
	      c4 = CubeToSphere(Coord3D( vchunk->xmin,             vchunk->ymin+b+GRID_STEP,           vchunk->zmin+a));
	      tax = 1.0f - a / width;
	      tbx = 1.0f - (a+GRID_STEP)/width;
	      tay = 1.0f - b / width;
	      tby = 1.0f - (b+GRID_STEP)/width;
	      
	   }
	 else if ( vchunk->node.GetPrefix() == 'N' || vchunk->node.GetPrefix() == 'S' )
	   { 
	      c1 = CubeToSphere(Coord3D( vchunk->xmin+a,           vchunk->ymin,             vchunk->zmin+b    ));
	      c2 = CubeToSphere(Coord3D( vchunk->xmin+a+GRID_STEP, vchunk->ymin,             vchunk->zmin+b    ));
	      c3 = CubeToSphere(Coord3D( vchunk->xmin+a+GRID_STEP, vchunk->ymin,             vchunk->zmin+b+GRID_STEP));
	      c4 = CubeToSphere(Coord3D( vchunk->xmin+a,           vchunk->ymin,             vchunk->zmin+b+GRID_STEP));
	      tax = a / width;
	      tbx = (a+GRID_STEP)/width;
	      tay = b / width;
	      tby = (b+GRID_STEP)/width;
	   }
	 else // Front or Back
	   {
	      c1 = CubeToSphere(Coord3D( vchunk->xmin+a,           vchunk->ymin+b,           vchunk->zmin    ));
	      c2 = CubeToSphere(Coord3D( vchunk->xmin+a+GRID_STEP, vchunk->ymin+b,           vchunk->zmin    ));
	      c3 = CubeToSphere(Coord3D( vchunk->xmin+a+GRID_STEP, vchunk->ymin+b+GRID_STEP, vchunk->zmin    ));
	      c4 = CubeToSphere(Coord3D( vchunk->xmin+a,           vchunk->ymin+b+GRID_STEP, vchunk->zmin    ));
	      tax = a / width;
	      tbx = (a+GRID_STEP)/width;
	      tay = 1.0f - b / width;
	      tby = 1.0f - (b+GRID_STEP)/width;
	   }	  
	 
	 tay = 1.0f - tay;
	 tby = 1.0f - tby;
*/	 

#endif /* USE_PRECOMPUTED_GRID not defined */

	  if ( !options.lighting)
	    {   
	       glVertex3fv( c1.GetCoords());
	       glVertex3fv( c2.GetCoords());
	       glVertex3fv( c3.GetCoords());
	       glVertex3fv( c4.GetCoords());	
	    }
	  else
	    {
	       glNormal3fv( c1.GetCoords()); glVertex3fv( c1.GetCoords());
	       glNormal3fv( c2.GetCoords()); glVertex3fv( c2.GetCoords());
	       glNormal3fv( c3.GetCoords()); glVertex3fv( c3.GetCoords());
	       glNormal3fv( c4.GetCoords()); glVertex3fv( c4.GetCoords());	
	    }
       }
   glEnd();
}

void Planet::DrawChunk( VirtualChunk *vchunk, float normalized_screen_space)
{     	    
   
   Chunk *chunk = cache->GetChunk( vchunk->node);
       
   if ( chunk == NULL)
     {
	cache->PlanChunkLoading( vchunk->node);
	DrawChunkOcean( vchunk);
     }
   else
     {
	chunk->texture->Bind();
	glEnable( GL_TEXTURE_2D);
	glDisable( GL_BLEND);
	glColor3f( 1.0f, 1.0f, 1.0f);	
	chunk->Draw();
     
	// Test if we are near to render children.
	// If so preload them so as to achieve continuous switches
   	
	if ( chunk != NULL && !chunk->children_preloaded && vchunk->node.GetDepth() < nb_lod && normalized_screen_space > 0.8f)
	  {
	     cache->PlanChunkLoading( vchunk->GetChild1().node);
	     cache->PlanChunkLoading( vchunk->GetChild2().node);
	     cache->PlanChunkLoading( vchunk->GetChild3().node);
	     cache->PlanChunkLoading( vchunk->GetChild4().node);
	     
	     // Remember that preloading have been done to avoid filling
	     // the queue of chunks to be loaded //TODO: do it progressively
	     
	     chunk->children_preloaded = true;
	  }
	
	cache->ReleaseChunk( chunk);
     }
   
#ifdef _DEBUG
   
   if ( options.bounding_box)
     {
	// Draw bounding sphere
	 
	float bs_ray = RADIUS / (float)( 1 << vchunk->node.GetDepth());	
		
	Vector3f center = grids[ vchunk->node.GetFace()].Get( (vchunk->min_gx+vchunk->max_gx)/2, (vchunk->min_gy+vchunk->max_gy)/2);

	glPushMatrix();
	glTranslatef( center[0], center[1], center[2]);
	 
	GLUquadricObj *sphereObj = gluNewQuadric();
	gluSphere( sphereObj, bs_ray, 8, 8);
	
	glPopMatrix();
     }
   
   if ( options.subdivisions)
     {
	// Draw quadtree subdivisions
    	    
	int grid_number = vchunk->node.GetFace();
	Vector3f c1, c2;
	glLineWidth( 2.0f);
	glBegin( GL_LINES);
	glColor3f( normalized_screen_space, 1.0f-normalized_screen_space, 0.0f);
	
	for( unsigned int gy = vchunk->min_gy ; gy < vchunk->max_gy ; gy++)
	  {
	     c1 = grids[grid_number].Get( vchunk->min_gx,   gy  );
	     c2 = grids[grid_number].Get( vchunk->min_gx,   gy+1  );
	     glVertex3fv( c1.GetCoords());
	     glVertex3fv( c2.GetCoords());
	     c1 = grids[grid_number].Get( vchunk->max_gx,   gy  );
	     c2 = grids[grid_number].Get( vchunk->max_gx,   gy+1  );
	     glVertex3fv( c1.GetCoords());
	     glVertex3fv( c2.GetCoords());
	  }
	
	for( unsigned int gx = vchunk->min_gx ; gx < vchunk->max_gx ; gx++)
	  {    
	     c1 = grids[grid_number].Get( gx,               vchunk->min_gy  );
	     c2 = grids[grid_number].Get( gx+1,             vchunk->min_gy  );
	     glVertex3fv( c1.GetCoords());
	     glVertex3fv( c2.GetCoords());
	     c1 = grids[grid_number].Get( gx,               vchunk->max_gy  );
	     c2 = grids[grid_number].Get( gx+1,             vchunk->max_gy  );
	     glVertex3fv( c1.GetCoords());
	     glVertex3fv( c2.GetCoords());
	  }
	
	glEnd();
	glColor3f( 1.0f, 1.0f, 1.0f);
	glLineWidth( 1.0f);	
     }
   
#endif /* _DEBUG */       
   
}

VirtualChunk Planet::GetVirtualChunk( Node node)
{
   unsigned int base_width = ( grid_width >> node.GetDepth());
   unsigned int min_gx = base_width * node.GetPositionX();
   unsigned int max_gx = min_gx + base_width;
   unsigned int min_gy = base_width * node.GetPositionZ();
   unsigned int max_gy = min_gy + base_width;

   return VirtualChunk( min_gx, max_gx, min_gy, max_gy, node);
}

   
void Planet::Draw( Camera *camera)
{
   // Reset counters
   drawn_nodes_counter = 0;
   explored_nodes_counter = 0;
   
   VirtualChunk vchunk;

   // Draw front
   vchunk = VirtualChunk( 0, grid_width, 0, grid_width, Node( 0,0,0, Cube::Front));
   DrawSurface( &vchunk, camera);

   // Draw east
   vchunk = VirtualChunk( 0, grid_width, 0, grid_width, Node( 0,0,0, Cube::East));
   DrawSurface( &vchunk, camera);

   // Draw back
   vchunk = VirtualChunk( 0, grid_width, 0, grid_width, Node( 0,0,0, Cube::Back));
   DrawSurface( &vchunk, camera);
   
   // Draw west
   vchunk = VirtualChunk( 0, grid_width, 0, grid_width, Node( 0,0,0, Cube::West));
   DrawSurface( &vchunk, camera);
   
   // Draw north
   vchunk = VirtualChunk( 0, grid_width, 0, grid_width, Node( 0,0,0, Cube::North));
   DrawSurface( &vchunk, camera);
   
   // Draw south
   vchunk = VirtualChunk( 0, grid_width, 0, grid_width, Node( 0,0,0, Cube::South));
   DrawSurface( &vchunk, camera);   
}


void Planet::DrawMeridian( float lon_deg, float step_deg)
{
   glBegin( GL_LINES);
   
   for ( float lat_deg = -90.0f ; lat_deg < 90.0f ; lat_deg += step_deg)
     {
	glVertex3fv( PolarToCoord3f( lat_deg,          lon_deg, RADIUS+EPSILON).GetCoords());
	glVertex3fv( PolarToCoord3f( lat_deg+step_deg, lon_deg, RADIUS+EPSILON).GetCoords());	
     }
   
   glEnd();
}

void Planet::DrawLatitude( float lat_deg, float step_deg)
{
   glBegin( GL_LINES);
   
   for( float lon_deg = 0.0f ; lon_deg < 360.0f ; lon_deg += step_deg)
     {
	glVertex3fv( PolarToCoord3f( lat_deg, lon_deg,          RADIUS+EPSILON).GetCoords());
	glVertex3fv( PolarToCoord3f( lat_deg, lon_deg+step_deg, RADIUS+EPSILON).GetCoords());	
     }
   
   glEnd();	
}
   
void Planet::DrawGrids()
{
   
   glDisable( GL_TEXTURE_2D);
   
   for(int p=0;p<6;p++)
     {
	
	for( int gy=0;gy<32;gy++)
	  for( int gx=0;gx<32;gx++)
	    {
	       
	       glColor3f( (float)gy/32.0f, (float)gx/32.0f, 0.0f);
	       
	       glBegin( GL_LINE_LOOP);
	       
	       glVertex3fv( grids[p].Get(gx, gy).GetCoords());
	       glVertex3fv( grids[p].Get(gx+1, gy).GetCoords());
	       glVertex3fv( grids[p].Get(gx+1, gy+1).GetCoords());
	       glVertex3fv( grids[p].Get(gx, gy+1).GetCoords());
	       
	       glEnd();
	       
	    }	
     }
}


void Planet::PickCoordinates( int screen_x, int screen_y, float *lat_deg, float *lon_deg)
{
   double modelMatrix[16];
   double projMatrix[16];
   GLint viewPort[4];
   double px, py, pz;
   double wx, wy, wz;
   float screen_z;
   
   // Read depth value of the pixel under the picked pixel 
   
   glReadPixels( screen_x, screen_y, 1, 1, GL_DEPTH_COMPONENT, GL_FLOAT, &screen_z);
   
   // Get all transformation matrix to reverse projection
  
   glGetDoublev( GL_MODELVIEW_MATRIX, modelMatrix ) ;
   glGetDoublev( GL_PROJECTION_MATRIX, projMatrix ) ;
   glGetIntegerv( GL_VIEWPORT, viewPort) ;
   
   px = (double)screen_x;
   py = (double)( viewPort[3] - (GLint) screen_y - 1);
   pz = (double)screen_z;
   
   // Unproject
   
   gluUnProject ( px, py, pz,
		  modelMatrix, projMatrix, viewPort,
		  &wx, &wy, &wz);
   
   // Now wx, wy, wz correspond to pixel picked by mouse in object world coordinates

   // Check that the cursor is over the planet 
   // Can have used index instead of this test
   
   float dist = sqrtf( wx*wx + wy*wy + wz*wz);
   
   if ( dist-EPSILON < radius && dist+EPSILON > radius)
     {	
	Coord3fToPolar( Vector3f( wx, wy, wz), lat_deg, lon_deg);
	
	// Convert from radians to degrees
   	
	*lat_deg *= 180.0f / M_PI;
	*lon_deg *= 180.0f / M_PI;
     }
   else
     {
     	*lat_deg = NAN;
	*lon_deg = NAN;
     }
}

unsigned int Planet::GetDrawnNodes()
{
   return drawn_nodes_counter;
}
   
unsigned int Planet::GetExploredNodes()
{
   return explored_nodes_counter;
}
