#ifndef _OBSERVER_HPP
#define _OBSERVER_HPP

#include <SDL.h>
#include "math/Vector.hpp"

class Camera;

/**
 * Abstract Camera class
 * 
 */

class Observer
{
   
 public:
   
   Observer(){};
   virtual ~Observer(){};
      
   virtual void UpdateCamera( Camera *camera) = 0;
   virtual void OnSDLEvent( SDL_Event *event){}
   virtual void Update() = 0;
};

/**
 * Position and orient the camera to have a look at the planet 
 * centered on specified latitude and longitude   
 * so that north will always be at the top, south at the bottom 
 * east on the left and east on the right.
 * @param distance distance from origin in OpenGL units
 * @param lat_deg latitude to center the view on in degrees
 * @param lon_deg longitude to center the view on in degrees
 */

class Observer_Ortho : public Observer
{
   
 public:
   
   Observer_Ortho( float min_dist, float max_dist, float initial_dist, float initial_lon_deg, float initial_lat_deg);
   ~Observer_Ortho();
 
   void MoveOnLatitude( float relative_motion_deg);
   void MoveOnLongitude( float relative_motion_deg);
   void MoveOnDistance( float relative_motion);
   void UpdateCamera( Camera *camera);
   void Update();
   void OnSDLEvent( SDL_Event *event);

 private:
   
   float min_distance;
   float max_distance;
   float cur_distance;
   float lon_deg;
   float lat_deg;
   bool auto_rotate;
};
 
class Observer_Quaternion : public Observer
{
 public:
   
   Observer_Quaternion();
   ~Observer_Quaternion();

   void UpdateCamera( Camera *camera);
   void Update();
   void OnSDLEvent( SDL_Event *event);
};

class Observer_SpaceShip : public Observer
{
 
 public:
 
   Observer_SpaceShip( float min_dist, float max_dist, Vector3f initial_position);
   ~Observer_SpaceShip();

   void SetHeading( float heading_deg);
   void MoveHeading( float heading_motion_deg);
   void SetPitch( float pitch_deg);
   void MovePitch( float pitch_motion_deg); 
   void MoveForward( float norm);
   void MoveBackward( float norm);
   
   void UpdateCamera( Camera *camera);
   void Update();
   void OnSDLEvent( SDL_Event *event);

 private:
   
   // No latitude, longitude and altitude (like a plane) as a spaceship is out
   // of earth grvity
   Vector3f position;
   float heading_deg;
   float pitch_deg;
   float roll_deg; // stay always zero
   float min_distance;
   float max_distance;
};

#endif /* _OBSERVER_HPP */
