#include <SDL.h>
#include "Observer.hpp"
#include "Camera.hpp"
#include "projection.hpp"

/* == Observer_Ortho implemenation ========================================= */

Observer_Ortho::Observer_Ortho( float min_dist, float max_dist, float initial_dist, float initial_lon_deg, float initial_lat_deg) 
{
   min_distance = min_dist;
   max_distance = max_dist;
   cur_distance = initial_dist;
   lon_deg = initial_lon_deg;
   lat_deg = initial_lat_deg;
   auto_rotate = false;
}

Observer_Ortho::~Observer_Ortho()
{
}

void Observer_Ortho::MoveOnLongitude( float relative_motion_deg)
{
   lon_deg += relative_motion_deg;
}

void Observer_Ortho::MoveOnLatitude( float relative_motion_deg)
{
   lat_deg += relative_motion_deg;
   
   if ( lat_deg > 90.0f )
     lat_deg = 90.0f;
   else if ( lat_deg < -90.0f)
     lat_deg = -90.0f;
}

void Observer_Ortho::MoveOnDistance( float relative_motion)
{
   cur_distance += relative_motion;
   
   if ( cur_distance < min_distance )
     cur_distance = min_distance;
   else if ( cur_distance > max_distance )
     cur_distance = max_distance;
}

void Observer_Ortho::Update()
{
   Uint8* keymap ;
   keymap = SDL_GetKeyState( NULL) ;
   bool turbo = SDL_GetModState() & KMOD_SHIFT; 
   float speed = 1.0f * (cur_distance-min_distance/2.0f)/max_distance;
   
   if ( keymap[SDLK_PAGEUP])
     MoveOnDistance(  - ( turbo ? 0.004f : 0.002f));
   if ( keymap[SDLK_PAGEDOWN])
     MoveOnDistance(  + ( turbo ? 0.004f : 0.002f));
   if ( keymap[SDLK_LEFT])
     MoveOnLongitude(  - ( turbo ? 0.2f : 0.1f ));
   if ( keymap[SDLK_RIGHT])
     MoveOnLongitude(  + ( turbo ? 0.2f : 0.1f ));
   if ( keymap[SDLK_UP])
     MoveOnLatitude( + ( turbo ? 0.2f : 0.1f ));
   if ( keymap[SDLK_DOWN])
     MoveOnLatitude( - ( turbo ? 0.2f : 0.1f ));
   if ( auto_rotate)
     MoveOnLongitude( speed);
}

void Observer_Ortho::OnSDLEvent( SDL_Event *event)
{
   bool turbo = SDL_GetModState() & KMOD_SHIFT; 

   // Process Mouse events
   
   if ( event->type == SDL_MOUSEBUTTONDOWN && 
	event->button.button == SDL_BUTTON_WHEELDOWN)
     {
	MoveOnDistance( +0.005f * (cur_distance/min_distance));
     }
   else if ( event->type == SDL_MOUSEBUTTONDOWN &&
	     event->button.button == SDL_BUTTON_WHEELUP)
     {
	MoveOnDistance( -0.005f * (cur_distance/min_distance));
     }
   else if ( event->type == SDL_MOUSEMOTION &&
	     ( event->motion.state & SDL_BUTTON_LMASK))
     {
	MoveOnLatitude(((float)event->motion.yrel) * (1.0f*(cur_distance-min_distance/2.0f)/max_distance));
	MoveOnLongitude(-((float)event->motion.xrel) * (1.0f*(cur_distance-min_distance/2.0f)/max_distance));
     }
   else if ( event->type == SDL_MOUSEMOTION && ( event->motion.state & SDL_BUTTON_RMASK))
     {
	MoveOnDistance( (float)event->motion.yrel / 2000.0f);
     }
   
   // Process Keyboard events
   
   if ( event->type == SDL_KEYDOWN)
     {
	switch (event->key.keysym.sym)
	  {
	   case SDLK_r: auto_rotate = !auto_rotate; break;
	   default: break;
	  }
     }
}

void Observer_Ortho::UpdateCamera( Camera *camera)
{
   camera->SetEyePosition( PolarToCoord3f( lat_deg, lon_deg, cur_distance));
   camera->SetLook( Vector3f( 0.0f, 0.0f, 0.0f));
   camera->SetUp( Vector3f( 0.0f, 1.0f, 0.0f));		    
}


/* === Observer_Quaternion implementation ============================================ */

Observer_Quaternion::Observer_Quaternion()
{
}

Observer_Quaternion::~Observer_Quaternion()
{
}

void Observer_Quaternion::UpdateCamera( Camera *camera)
{
   
}

void Observer_Quaternion::Update()
{
}

void Observer_Quaternion::OnSDLEvent( SDL_Event *event)
{
}


/* === Observer_SpaceShip implementation ============================================= */

Observer_SpaceShip::Observer_SpaceShip( float min_dist, float max_dist, Vector3f initial_position)
{
   min_distance = min_dist;
   max_distance = max_dist;
   position = initial_position;
}

Observer_SpaceShip::~Observer_SpaceShip()
{
}

void Observer_SpaceShip::SetHeading( float heading_deg)
{
   this->heading_deg = heading_deg;
}

void Observer_SpaceShip::MoveHeading( float heading_motion_deg)
{
   heading_deg += heading_motion_deg;
}

void Observer_SpaceShip::SetPitch( float new_pitch_deg)
{
   this->pitch_deg = new_pitch_deg;
   while ( pitch_deg >= 180.0f)
     pitch_deg -= 360.0f;
   while ( pitch_deg < -180.0f)
     pitch_deg += 360.0f;
}

void Observer_SpaceShip::MovePitch( float pitch_motion_deg)
{
   pitch_deg += pitch_motion_deg;
   while ( pitch_deg >= 180.0f)
     pitch_deg -= 360.0f;
   while ( pitch_deg < -180.0f)
     pitch_deg += 360.0f;
}

void Observer_SpaceShip::MoveForward( float norm)
{
   position[0] += norm * sinf( heading_deg*M_PI/180.0f)*cosf(pitch_deg*M_PI/180.0f);
   position[1] += norm * sinf( pitch_deg*M_PI/180.0f);
   position[2] += norm * cosf( pitch_deg*M_PI/180.0f)*cosf( heading_deg*M_PI/180.0f);
}

void Observer_SpaceShip::MoveBackward( float norm)
{
   MoveForward( -norm);
}

void Observer_SpaceShip::UpdateCamera( Camera *camera)
{
   camera->SetEyePosition( position);   
   
   Vector3f look( sinf(heading_deg*M_PI/180.0f)*cosf(pitch_deg*M_PI/180.0f),
		  sinf(pitch_deg*M_PI/180.0f),
		  cosf(heading_deg*M_PI/180.0f)*cosf(pitch_deg*M_PI/180.0f));
   
   Vector3f up( sinf(heading_deg*M_PI/180.0f)*cosf((pitch_deg+90.0f)*M_PI/180.0f),
		sinf((pitch_deg+90.0f)*M_PI/180.0f),
		cosf(heading_deg*M_PI/180.0f)*cosf((pitch_deg+90.0f)*M_PI/180.0f));
			     
   camera->SetLook( position + look);
   
   camera->SetUp( up);//Vector3f( 0.0f, 1.0f , 0.0f));

   
   fprintf( stderr, "\nspacehip: %d %d\n", (int)heading_deg, (int)pitch_deg);
   
}

void Observer_SpaceShip::Update()
{
   Uint8* keymap ;
   keymap = SDL_GetKeyState( NULL) ;
   bool turbo = SDL_GetModState() & KMOD_SHIFT; 
   float cur_distance = position.distance( Vector3f( 0.0f, 0.0f, 0.0f));
   float speed = 1.0f * (cur_distance-min_distance/2.0f)/max_distance;
     
   if ( keymap[SDLK_UP])
     MoveForward(  speed*( turbo ? 0.2f : 0.1f ));
   if ( keymap[SDLK_DOWN])
     MoveBackward( speed*( turbo ? 0.2f : 0.1f ));
   if ( keymap[SDLK_LEFT])
     MoveHeading(  + ( turbo ? 1.0f : 2.0f ));
   if ( keymap[SDLK_RIGHT])
     MoveHeading(  - ( turbo ? 1.0f : 2.0f ));  
   if ( keymap[SDLK_PAGEUP])
     MovePitch(    - ( turbo ? 1.0f : 2.0f ));
   if ( keymap[SDLK_PAGEDOWN])
     MovePitch(  + ( turbo ? 1.0f : 2.0f ));

}

void Observer_SpaceShip::OnSDLEvent( SDL_Event *event)
{
   bool turbo = SDL_GetModState() & KMOD_SHIFT; 

   // Process Mouse events
   
   if ( event->type == SDL_MOUSEMOTION )
     {
	MoveHeading(  (float)event->motion.xrel * ((pitch_deg>=0.0f)?-1.0f:1.0f) );
	MovePitch(   -(float)event->motion.yrel);
     }
   
   // Process Keyboard events

   float cur_distance = position.distance( Vector3f( 0.0f, 0.0f, 0.0f));
   float speed = 10.0f * (cur_distance-min_distance/2.0f)/max_distance;
   
   if ( event->type == SDL_KEYDOWN)
     {
	switch (event->key.keysym.sym)
	  {
	   default: break;
	  }
     }
}
