#ifndef _NODE_HPP
#define _NODE_HPP

/** \file Node.hpp
 * A class and associated methods to manage world scenery tiling with
 * Level of Details (LOD).
 */

#include <string>

using namespace std;

#include "projection.hpp"

/**
 * A class to enumerate the quadtree and explore it.                           
 * The quadtree is enumerate this way:
 * level 0 (root)
 *   nodes: 0
 * level 1
 *   nodes: 1 2 3 4
 *    .
 *    .
 * level n
 *   nodes: 2 exp ( n + 1 ) to 2 exp ( n + 2 ) -1  
 * 
 * Notes about limitations:                                           
 * As we currently use unsigned long as type for node identifier,     
 * total node count is limited to (2 power 32) 4Go nodes so we are limited
 * to 12 levels of details for any quadtree.
 */

class Node
{  
   
 public:
   
   /**
    * Default constructor, create front face root node
    */
   inline Node() {level=0;x=0;z=0;face=Cube::Front;}
   
   /**
    * Copy contructor.
    * @param node object to be copied
    */
   inline Node(const Node &node) { this->level = node.level; this->x=node.x;this->z=node.z;this->face=node.face;}

   /**
    * Construct a node from its logical coordinates.
    * @param level of detail
    * @param x logical x coordinate at the given level
    * @param z logical z coordinate at the given level
    * @param face of the projection cube
    */
   inline Node( const unsigned short level, const unsigned long x, const unsigned long z, const Cube::Face face) { this->level=level; this->x=x; this->z=z; this->face=face;}
   
   /**
    * Construct a node from its identifier.
    * @param node_number identifier of the node
    */
   Node( const unsigned long node_number, const Cube::Face face);
  
   /** 
    * Dstructor
    */   
   inline ~Node(){}
   
   // Accessors for logical coordinates
   
   /**
    *@return x logical location 
    */
   inline unsigned long GetPositionX() const {return x;}
   
   /**
    *@return z logical location 
    */
   inline unsigned long GetPositionZ() const {return z;}
   
   /**
    *@return node's level of detail 
    */
   inline unsigned long GetDepth() const {return level;} 
   
   /**
    *@return unique identifier for this node 
    */
   inline unsigned int GetNumber() const {return GetNodesBeforeLevel(level) + ( z << level ) + x;}

   /**
    *@return unique identifier for this node 
    */
   inline char GetPrefix() const {return Cube::FacePrefix[face];}

   /**
    *@return unique identifier for this node 
    */
   inline Cube::Face GetFace() const {return face;}

   /**
    *@return unsique identifier as string
    */
   string *GetID( const string &prefix) const;
  
   // Exploration between Levels of Detail
   
   /**
    *@return father node identifier
    */
   inline Node GetFather() const {return Node((unsigned short)(level-1), (unsigned long)(x/2), (unsigned long)(z/2), face);}
   
   /**
    * Given a child id, return child node
    * @param child can be:
    *  0 - child of North West
    *  1 - child of North East
    *  2 - child of South West
    *  3 - child of South East
    *@return child node
    */
   inline Node GetChild( const int child_id) const {return Node(level+1, x*2+(child_id%2), z*2+(child_id/2), face);}
   
   /**
    * Given a child_id, return one of the 16 children and children         
    * neighboors identifier or 0 if none available (possible at boundaries)
    *                                                                      
    * Note: when none is available, 0 is safe because it can't be a valid  
    * child since it's the root identifier.                     
    */
   unsigned long GetChild16( const long x, const long z) const;
      
 private:
   inline unsigned long GetNodesBeforeLevel( const int level) const { return 0x55555555 & ( 1 << (level*2) ) - 1 ;}
   
   unsigned long node_number;
   unsigned short level;
   unsigned long x;
   unsigned long z;
   Cube::Face face;
   
 public:
   inline bool operator == ( Node &n)
     {
	return GetNumber() ==  n.GetNumber() && face == n.face;
     }

   inline bool operator != ( Node &n)
     {
	return GetNumber() !=  n.GetNumber() || face != n.face;;
	
     }

};


/*                                                                      
 *  GetChild16 table:                                        
 *                                                                      
 *                                (-1,-1)   ( 0,-1)   ( 1,-1)   ( 2,-1)
 *
 *                                        +---------+---------+
 *                                        |         |         |
 *                                (-1, 0) | ( 0, 0) | ( 1, 0) | ( 2, 0)
 *                                        |         |         |        
 *                                        +---------+---------+        
 *                                        |         |         | 
 *                                (-1, 1) | ( 0, 1) | ( 1, 1) | ( 2, 1)
 *                                        |         |         |   
 *                                        +---------+---------+   
 * 
 *                                (-1, 2)   ( 0, 2)   ( 1, 2)   ( 2, 2)
 *
 */

#endif
