#ifndef _DATA_SET_HPP
#define _DATA_SET_HPP

/** \file DataSet.hpp
 * The purpose of this file is to provide classes to handle large data sets
 * avoiding using a lot of small files by packing them into one only big file. 
 *
 * This way we can manage:
 * 1) Save space when a dataset is made of a huge number of small files 
 * ( space that could have been lost because of clusters or inodes min size)
 * 2) Don't be slow down by bad filesystem implementation when retrieving 
 * a file in a directory that contain a huge number of files ( this 
 * implementation use a hash table to retrieve files in a O(n)=log(n) time)
 */

#include <string>
#include <map>

using namespace std;

struct DataFile
{
   unsigned int offset_bytes;
   unsigned int length_bytes;
};

struct SDL_RWops;

/**
 * This class manage the DataSet. It can be used to create dataset (i.e. pack 
 * files) as well as to read the packaed files. 
 * 
 * Notes about limitations:
 * As we use a signed integer for file offset, the dataset file size can't
 * be more than 2GB
 */

class DataSet
{
 public:
   
   /**
    * Default constructor
    */
   DataSet();
   
  /**
   *Destructor
   */
   ~DataSet();
   
   /**
    * Open a dataset for reading
    * @param filename dataset to open (must be full qualified with path and extension)  
    * @exception raised on error
    **/
   void Open( string filename);

   /**
    * Save a DataSet
    * @param filename file into wich the dataset is to be saved (must be full qualified with path and extension)  
    * @exception raised on error
    **/
   void Save( string filename);

   /**
    * Set the base path before packing the files ( use it only when you want
    * to create a dataset, this function have no effect when reading a dataset)
    * @param base_path base path from which the files will be loaded.
    * This path can be saw as the root directory of the dataset.
    * @exception raised on error
    **/
   void SetBasePath( string base_path);

   /**
    * Add a file in the dataset ( use it only when you create a dataset)
    * @param filename with path (related to base path directory, See SetBasePath())
    * @exception raised when the file can't be opened
    **/
   void AddFile( string filename);
   
   /**
    * Remove a file from a dataset ( use it only when you create a dataset)
    * @param filename with path (related to base path directory, See SetBasePath())
    **/
   void RemoveFile( string filename);

   /**
    * Open a file from the DataSet
    * @param filename with path
    * @return SDL file descriptor of the file ( that must be closed with 
    * SDL_RWclose)
    */
   SDL_RWops *OpenFile( string filename);
   
   /**
    * Print the list of the files packed in the dataset on standard output.
    */
   void DumpIndex();

   /**
    * Test dataset state
    * @return true if the dataset is opened
    */
   inline bool isReady() { return ready;}
   
 private:
   map< string, DataFile> index;
   SDL_RWops *file;
   unsigned int start_of_data;
   string base_path;
   bool ready;
};

#endif /* _DATA_SET_HPP */
