#include <iostream>
#include <SDL.h>
#include <SDL_endian.h>
//#ifdef _UNIX
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
//#endif /* _UNIX */
//#include "sdlzlib.h"
#include "DataSet.hpp"

#define BUFFER_SIZE 65535

struct SDL_RWops_hidden
{
   SDL_RWops *true_rw_ops;
   int offset_start;
   int offset_end;
};

DataSet::DataSet()
{
   ready = false;
}

DataSet::~DataSet()
{
}

void DataSet::Open( string filename)
{  

   file = SDL_RWFromFile( filename.c_str(), "rb");
   
   if ( file == NULL)
     {  
	cerr << "DataSet: can't open " << filename << endl;
	return;
     }
   
   unsigned int nb_entries = SDL_ReadLE32( file);
   unsigned int current_offset = 0;
   
   for( unsigned int entry = 0 ; entry < nb_entries ; entry++)
     {
	DataFile df;
	
	df.offset_bytes = SDL_ReadLE32( file);
	df.length_bytes = SDL_ReadLE32( file);
	unsigned short fname_length = SDL_ReadLE16( file);
	char fname[fname_length+1];
	SDL_RWread( file, fname, sizeof(char), fname_length);
	fname[fname_length] = '\0'; 
	df.offset_bytes = current_offset; 
	current_offset += df.length_bytes;
	index[ fname] = df;
     }

   start_of_data = SDL_RWtell( file);

   ready = true;
   
   cout << "DataSet: dataset " << filename << " ready (" <<  index.size() << " items)" << endl;
}

void DataSet::Save( string filename)
{
   // Open file in writing mode
   
   file = SDL_RWFromFile( filename.c_str(), "wb");
   
   if ( file == NULL)
     {  
	cerr << "DataSet: can't save to " << filename << endl;
	return;
     }

   // Write the index
  
   unsigned int nb_entries = index.size();
   
   SDL_WriteLE32( file, nb_entries);
   
   map<string, DataFile>::iterator current = index.begin();
   map<string, DataFile>::iterator last = index.end();
   
   while( current != last)
     {
	SDL_WriteLE32( file, current->second.offset_bytes);
	SDL_WriteLE32( file, current->second.length_bytes);
	SDL_WriteLE16( file, current->first.size());
	SDL_RWwrite( file, current->first.c_str(), sizeof( char), current->first.size()); // no need to save '\0'
	
	current++;
     }

   // Write all the data 
   
   current = index.begin();
   last = index.end();
   
   unsigned char buffer[BUFFER_SIZE];
   
   while( current != last)
     {
	
	// Open file to be packed
	
	SDL_RWops *f = SDL_RWFromFile( (base_path+current->first).c_str(), "rb");
	
	if ( f == NULL )
	  {
	     cerr << "DataSet: Can't open file " << current->first << ", it will not be in the data set!" << endl;   
	     current++;
	     continue;
	  }
	
	unsigned int buffer_felt = 0;
	unsigned int data_to_copy = current->second.length_bytes;
	
	while ( data_to_copy > 0)
	  {
	     buffer_felt = SDL_RWread( f, buffer, sizeof( char), (data_to_copy>BUFFER_SIZE)?BUFFER_SIZE:data_to_copy);
	     
	     data_to_copy -= buffer_felt;
	     
	     SDL_RWwrite( file, buffer, sizeof( char), buffer_felt);
	  }
	
	// Close file
	
	SDL_RWclose( f);
	
	current++;
     }
   
   SDL_RWclose( file);
   
   cout << "DataSet: " << nb_entries << " entries written to " << filename << endl;
}

//#ifdef _UNIX

void DataSet::SetBasePath( string base_path)
{
   this->base_path = base_path;
}

void DataSet::AddFile( string filename)
{
   DataFile df;
   struct stat st;
   
   if ( stat( (base_path+filename).c_str(), &st) != 0 )
     {
	cerr << "DataSet: can't stat " << filename << ", it will not be included" << endl;
	return;
     }
   
   df.length_bytes = st.st_size;
     
   index[filename] = df;

   cout << "DataSet: added " << filename << " (" << df.length_bytes << " bytes)" << endl; 
}

//#endif

void DataSet::DumpIndex()
{
   map<string, DataFile>::iterator current = index.begin();
   map<string, DataFile>::iterator last = index.end();

   while( current != last)
     {
	cout << (*current).first << " offset " << (*current).second.offset_bytes << " length " << (*current).second.length_bytes <<endl; 
	current++;
     }
   
}

static int ds_wrapper_sdl_rw_seek(SDL_RWops *context, int offset, int whence)
{
   SDL_RWops_hidden *h_ops = (SDL_RWops_hidden *) context->hidden.unknown.data1;

   int true_position;
   
   switch( whence)
     {
      case SEEK_SET:
	true_position = h_ops->offset_start + offset; 
	break;
	
      case SEEK_CUR:
        true_position = SDL_RWtell( h_ops->true_rw_ops) + offset;
	break;
	
      case SEEK_END:
	true_position = h_ops->offset_end - offset;
	break;
	
      default:
	//SDL_Error( SDL_EFSEEK); // can't compile for win32
	return -1;
     }
   
   if ( true_position >= h_ops->offset_start && true_position < h_ops->offset_end)
     {
	return SDL_RWseek( h_ops->true_rw_ops, true_position, SEEK_SET) - h_ops->offset_start;
     }
   else
     {
	//SDL_Error( SDL_EFSEEK); // TODO: can't compile for win32
	return -1;
     }
}

static int ds_wrapper_sdl_rw_read( SDL_RWops *context, void *ptr, int size, int maxnum)
{  
   int nread;
   int nb_blocks_to_read;
   
   SDL_RWops_hidden *h_ops = (SDL_RWops_hidden *) context->hidden.unknown.data1;
   
   int cur_pos_bytes = SDL_RWtell( h_ops->true_rw_ops);
      
   if ( cur_pos_bytes >= h_ops->offset_end )
     {
	return 0;
     }
   
   if ( maxnum*size >= h_ops->offset_end - cur_pos_bytes)
     {
	nb_blocks_to_read = ( h_ops->offset_end-1 - cur_pos_bytes ) / size;
     }
   else
     nb_blocks_to_read = maxnum;
   
   nread = SDL_RWread( h_ops->true_rw_ops, ptr, size, nb_blocks_to_read);

   if ( nread < 0 ) 
     {	
	//SDL_Error( SDL_EFREAD); TODO don't compile for win32
	return 0;
     }

   return nread;
}

static int ds_wrapper_sdl_rw_write(SDL_RWops *context, const void *ptr, int size, int num)
{
   throw string( "Sorry, ds_wrapper_sdl_rw_write not implemennted yet !");
   
   return 0;
}

static int ds_wrapper_sdl_rw_close(SDL_RWops *context)
{
   SDL_RWops_hidden *h_ops = (SDL_RWops_hidden *) context->hidden.unknown.data1;
 
   delete h_ops;
   SDL_FreeRW( context);
 
   return 0;
}


SDL_RWops *DataSet::OpenFile( string filename)
{
   map< string, DataFile>::iterator df_iterator;
   
   df_iterator = index.find( filename);
   
   if ( df_iterator != index.end())
     {	
	DataFile df = df_iterator->second; 

	SDL_RWops *ops = SDL_AllocRW();
	if (ops != NULL) 
	  {   
	     ops->seek  = ds_wrapper_sdl_rw_seek;
	     ops->read  = ds_wrapper_sdl_rw_read;
	     ops->write = ds_wrapper_sdl_rw_write;
	     ops->close = ds_wrapper_sdl_rw_close;
	     ops->hidden.unknown.data1 =  new SDL_RWops_hidden;
	     ((SDL_RWops_hidden *)ops->hidden.unknown.data1)->true_rw_ops  = file;
	     ((SDL_RWops_hidden *)ops->hidden.unknown.data1)->offset_start = start_of_data+df.offset_bytes;
	     ((SDL_RWops_hidden *)ops->hidden.unknown.data1)->offset_end   = start_of_data+df.offset_bytes+df.length_bytes;
	     
	     SDL_RWseek( file, start_of_data+df.offset_bytes ,SEEK_SET);
	     return ops;
	  }
	else
	  {
	     throw string("DataSet: can't allocate file descriptor");
	     return NULL;
	  }
     }
   else
     {
	throw string("DataSet: can't open ")+filename;
	return NULL;
     }
}
