#include <iostream>
#include <string>
#include <iterator>
#include <SDL.h>
#include "Cache.hpp"
#include "Chunk.hpp"
#include "Node.hpp"

#define MAX_ENTRIES            512
#define MAX_UNSIGNED_INT       ((unsigned int)-1)
#define CACHE_QUEUE_SIZE       32

int start_loading_thread( void *ct)
{
   ((Cache*)ct)->Manage();
   return 0;
}

Cache::Cache( unsigned int nb_entries)
{
   this->nb_entries = ( nb_entries < MAX_ENTRIES ) ? nb_entries : MAX_ENTRIES;
   stop_loading_thread = true;
   loading_thread = NULL;
   multi_thread = true;
}

Cache::~Cache()
{
   if ( loading_thread)
     {
        stop_loading_thread = true;
	SDL_SemPost( sem_something_to_load); // signal to ManageChunks that something happens
	cout <<  "Waiting for Cache thread to stop...";
	cout.flush();
	SDL_WaitThread( loading_thread, NULL);
	cout << "Done" << endl;
     }
   
   FreeAllEntries();
   
   for( unsigned int e=0;e<nb_entries;e++)
     SDL_DestroyMutex( entries[e].mutex);
   
   SDL_DestroySemaphore( sem_something_to_load);
   SDL_DestroyMutex( mutex_queue);
   
   delete []entries;
   delete []buzzy_entries;
}

void Cache::Initialize( Planet *planet)
{
   entries = new ChunkEntry[nb_entries];
   buzzy_entries = new bool[nb_entries];
   
   for(unsigned int e=0;e<nb_entries;e++)
     {
	entries[e].mutex = SDL_CreateMutex();
	buzzy_entries[e] = false;
     }
   
   this->planet = planet;
   
   mutex_queue = SDL_CreateMutex();
   stop_loading_thread = false;
   sem_something_to_load = SDL_CreateSemaphore( 0);
   if ( multi_thread)
     loading_thread = SDL_CreateThread( start_loading_thread, this);
}

void Cache::PlanChunkLoading( Node node)
{
   if ( multi_thread)
     {	
	SDL_mutexP( mutex_queue);
		
	// Check this node isn't already loaded or in the queue
      
	if ( NodeLoadedOrPlaned(node) == false)
	  {
	      if ( nodes_planed_for_loading.size() < CACHE_QUEUE_SIZE) 
	       { 
#ifdef _DEBUG_VERBOSE
		  cerr << "Cache:PlanNodeLoading: plan loading node " << node << endl;
#endif
		  nodes_planed_for_loading.push_back( node);
		  SDL_SemPost( sem_something_to_load); // signal to ManageChunks that something is planed
	       }
	  }
#ifdef _DEBUG_VERBOSE
	else  
	  cerr << " Cache:PlanNodeLoadain: " << node << " already planed" << endl;
#endif
	
	SDL_mutexV( mutex_queue);
     }
   else
     {
	if ( !NodeLoadedOrPlaned( node))
	  LoadChunk( node);
     }
}

//                                                                    //
// This method wait for chnuks to be planed for loading               //
// When a chunk is demanded, it awake and call the loading method     //
// then update the queue.                                             //
//                                                                    //
// Note: this method take some system time (i/o in loading process)   //
//       So this is why it is executed in a separate thread.          //
//                                                                    //
void Cache::Manage()
{	   
   // loading thread main loop
   
   while( !stop_loading_thread)
     {	
	// Wait for something to be planed for loading
	
	if ( SDL_SemWait( sem_something_to_load) == -1)
	  cout << "Cache::Manage: nothing to load, timeout reached..." << endl;
	
#ifdef _DEBUG_VERBOSE
        cout << "Cache::Manage: awaking" << endl;;
#endif
 
	SDL_mutexP( mutex_queue);
	
	bool node_have_to_be_loaded = true;
	Node node_to_be_loaded;
	
	if ( !nodes_planed_for_loading.empty())
	  {
	     node_to_be_loaded = nodes_planed_for_loading.front();
	  }
	else
	  {
	     node_have_to_be_loaded = false;
	  }
	
	SDL_mutexV( mutex_queue);
	
	if ( node_have_to_be_loaded)
	  {
	     LoadChunk( node_to_be_loaded);
	     SDL_mutexP( mutex_queue);
	     nodes_planed_for_loading.pop_front();
	     SDL_mutexV( mutex_queue);
	  }
     }
}

//                                                                    //
// This method return a pointer on the chunk of the given node        //
// if demanded chunk isn't present in entries cahe, return NULL       //
//                                                                    //
Chunk *Cache::GetChunk( Node node)
{
   Chunk *chunk = NULL;
   
   for( unsigned int e=0;e<nb_entries&&chunk==NULL;e++)
     {
	if ( buzzy_entries[e] && entries[e].node == node )
	  {
	     SDL_mutexP( entries[e].mutex);
	     if ( buzzy_entries[e] == false || entries[e].node != node) // check chunk have not been unloaded between test ( 2 lines up) and SDL_mutexP (1 line up) 
	       {
		  SDL_mutexV( entries[e].mutex);
		  continue; //return NULL;
	       }
	     chunk = entries[e].chunk; 
	     entries[e].last_use = SDL_GetTicks();
	  }
     }   
   return chunk;
}

void Cache::ReleaseChunk( Node node)
{
   bool found = false;
   
   for( unsigned int e=0;e<nb_entries&&(!found);e++)
     {
	if ( buzzy_entries[e] && entries[e].node == node )
	  {
	     SDL_mutexV( entries[e].mutex);
	     return;
	     //found = true;
	  }
     }
}

void Cache::ReleaseChunk( Chunk* chunk)
{
   bool found = false;
   
   for( unsigned int e=0;e<nb_entries&&(!found);e++)
     {
	if ( buzzy_entries[e] && entries[e].chunk == chunk )
	  {
	     SDL_mutexV( entries[e].mutex);
	     //found = true;
	  }
     }
}

unsigned int Cache::GetFreeEntry()
{
   int free_entry = -1;
   int least_recently_used_entry = 0;
   unsigned int least_recently_used = MAX_UNSIGNED_INT;
   
   // 1) look for a free entry and keep least recently used entry
   
   for( unsigned int e=0;(e<nb_entries)&&(free_entry==-1);e++)
     {
	if ( buzzy_entries[e] == false )
	  {
	     free_entry = e;
	  }
	else if ( entries[e].last_use < least_recently_used)
	  {
	     least_recently_used = entries[e].last_use;
	     least_recently_used_entry = e;
	  }  
     }
   
   // 2) if all entries are buzzy, free the least recently used
   
   if ( free_entry == -1 )
     {
	FreeEntry( least_recently_used_entry);
	free_entry = least_recently_used_entry;
     }
   
   return (unsigned int)free_entry;
}

void Cache::FreeEntry( unsigned int entry)
{
   if ( buzzy_entries[entry])
     {
	SDL_mutexP( entries[entry].mutex);
	delete entries[entry].chunk;
	entries[entry].chunk = NULL;
	buzzy_entries[entry] = false;
#ifdef _DEBUG_VERBOSE
	cout << "Cache::FreeEntry: Chunk " << entries[entry].node << " removed (entry " << entry << ")" << endl; 
#endif
	SDL_mutexV( entries[entry].mutex);
     }

}

void Cache::FreeAllEntries()
{     
   for( unsigned int e=0;e<nb_entries;e++)
     {
	FreeEntry( e);
     }
}

bool Cache::NodeLoadedOrPlaned( Node node)
{
   // check if node is already in the queue
  
   deque<Node>::iterator q;
   
   for( q=nodes_planed_for_loading.begin();q!=nodes_planed_for_loading.end();q++)
     if ( *q == node) 
       return true;
   
   // check if node is already in cache
   
   for(unsigned int e=0;e<nb_entries;e++)
     if ( buzzy_entries[e] && entries[e].node == node)
       return true;
   
   // if we are here, the node isn't loaded nor in the queue
   
   return false;
}

//                                                                    //
// This method take a node and load the corresponding chunk from disk //
// do allocations and fill structures in an optimized fashion ready   //
// for real-time rendering.                                           //
//                                                                    //
// Note: this function take take a lot of time (because of i/o) and   //
//       should be called in parallel with rendering not to block it. //
//                                                                    //
void Cache::LoadChunk( Node node)
{
   /*if ( NodeLoadedOrPlaned( node))
     {
        fprintf( stderr, "Cache::LoadChunk: Why loading a chunk which is already loaded!!!\n");
	fprintf( stderr, "This should have been detected before!\n");
	return;
     }*/
   
   unsigned int entry = GetFreeEntry();
   
   SDL_mutexP( entries[entry].mutex);
   
   Chunk* c = new Chunk();
   
   c->Load( node, planet);
  
   // Update entries
  
   entries[entry].node = node;
   entries[entry].chunk = c;
   entries[entry].last_use = SDL_GetTicks();
   buzzy_entries[entry] = true;
   
   SDL_mutexV( entries[entry].mutex);

#ifdef _DEBUG_VERBOSE
   
   cout << "Cache::LoadChunk: Chunk " << node << " loaded" << endl;
   
#endif /* _DEBUG */

}
