#ifndef _BUFFEROBJECT_HPP
#define _BUFFEROBJECT_HPP

/** \file BufferObject.hpp
 * A class and associated methods to manage efficiently geometry 
 * data and that can take advantage of recent GL extensions as 
 * VertexBufferObject_EXT.
 */

#include "gl_stuff.hpp"

/**
 * A class that store data about GL objects and try to render them    
 * efficiently.                                                       
 * In theese objects, we can store:                                   
 *  + vertices                                                        
 *  + normals                                                         
 *  + texcoords                                                       
 *  + colors                                                          
 *                                                                    
 * this data is packed efficiently ( using stride) and on systems that
 * support VertexBufferObject_EXT extension ( since OpenGL 1.4), data 
 * is uploaded in video memory at the first rendering time and only   
 * need to be binded the next times ( avoiding AGP bottleneck)        
 */

class BufferObject
{
   
 public:

   /**
    * Construct object along its description
    * @param nb_vertices number of vertices the object is composed of
    * #param vertex_size number of floats used to represent a vertex
    * #param normal_size number of floats used to represent a notmal
    * #param texcoord_size number of floats used to represent a texcoord
    * #param color_size number of floats used to represent a color
    */
   BufferObject( const unsigned long nb_vertices, const unsigned short vertex_size, const unsigned short normal_size, const unsigned short texcoord_size, const unsigned short color_size);
   
   /**
    * Destructor
    */
   ~BufferObject();
   
   /**
    * Bind object or simply do glxxxxxxPointer if VertexBufferObject 
    * isn't supported
    */
   void Bind();
   
   /** 
    * UnBind object
    */
   void UnBind();

   /** 
    * Fill a vertex in the object
    * @param index index of the vertex to fill
    * @param coord coordinate of the vertex to fill 
    * @param value to fill
    */
   inline void SetVertex( const unsigned long index, const unsigned short coord, const float value)
     {
	buffer[index*total_vertex_size+coord] = value; 
     }

   /** 
    * Fill a normal in the object
    * @param index index of the normal to fill
    * @param coord coordinate of the normal to fill 
    * @param value to fill
    */
   inline void SetNormal( const unsigned long index, const unsigned short coord, const float value)
     {
	buffer[index*total_vertex_size+normal_offset+coord] = value; 
     }
  
   /** 
    * Fill a texture coordinate in the object
    * @param index index of the texture coordinate to fill
    * @param coord coordinate of the texture coordinate to fill 
    * @param value to fill
    */
   inline void SetTexCoord( const unsigned long index, const unsigned short coord, const float value)
     {
	buffer[index*total_vertex_size+texcoord_offset+coord] = value; 
     }
   
   /** 
    * Fill a color in the object
    * @param index index of the color to fill
    * @param coord coordinate of the color to fill 
    * @param value to fill
    */
   inline void SetColor( const unsigned long index, const unsigned short coord, const float value)
     {
	buffer[index*total_vertex_size+color_offset+coord] = value; 
     }

 private:
   float *buffer;
   unsigned long nb_vertices;
   GLuint buffer_id;
   unsigned short vertex_size;
   unsigned short normal_size;
   unsigned short texcoord_size;
   unsigned short color_size;
   unsigned short normal_offset;
   unsigned short texcoord_offset;
   unsigned short color_offset;
   unsigned short total_vertex_size;
   bool buffer_uploaded;
   
   void UpLoad();
};

#endif
