/*
 * Souris - Jean-Christophe Duberga <jc@duberga.net>
 * 
 * Under GPL v2
 * 
 */

#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf-xlib/gdk-pixbuf-xlib.h>
#include <X11/extensions/shape.h>
#include <X11/Intrinsic.h>
#include <X11/extensions/record.h>
#include <X11/Xlibint.h> // ;))
#include "config.h"

#define PIXMAP_WIDTH_PX             100
#define PIXMAP_HEIGHT_PX            60
#define true                        1
#define false                       0 

typedef struct Souris Souris;
typedef struct Sprite Sprite;
typedef struct XWindow XWindow;
typedef int bool;

struct XWindow {
   Display *	display;
   gint		screen;
   Window	window;
   Window	root;
   gint		x, y;
   guint	root_width, root_height;
   guint	border;
   guint	depth;
};

struct Sprite 
{
   Pixmap	pixmap;
   Pixmap	mask;
   guint	width, height;
   GC		gc;
};

struct Souris 
{
   XWindow *xwindow;
   XWindow *xwindow_thread;
   GMutex  *mutex;
   Sprite **clicks_sprites;
   guint  current_sprite;
   guint  x, y;
   bool   left_button_pressed;
   float  middle_button_pressed;
   float  right_button_pressed;
};

// Thanks Xnee 

typedef union 
{   
   unsigned char    type ;
   xEvent           event ;
   xResourceReq     req   ;
   xGenericReply    reply ;
   xError           error ;
   xConnSetupPrefix setup;
} XRecordDatum ;


typedef void (*callback_ptr)( XPointer, XRecordInterceptData *);

int x_error (Display *display, XErrorEvent *error);
void x_init (struct XWindow *xwindow, guint width, guint height);
void x_update_position ( Souris *souris);
void x_update_sprite ( Souris *souris);
gboolean x_event (GIOChannel *source, GIOCondition condition, gpointer data);
gboolean x_timeout (gpointer data);

Souris * souris_init ();
void souris_update (Souris *souris);

Sprite * sprite_init (XWindow *xwindow, char *filename);
void sprite_paint (XWindow *xwindow, Sprite *sprite);
void sprite_repaint (XWindow *xwindow, Sprite *sprite);

void on_expose ( Souris *souris);

void * events_watcher( void *data);

int main (int argc, char **argv);


int x_error (Display *display, XErrorEvent *error) 
{
   char msg[128];
   XGetErrorText (display, error->error_code, msg, 127);
   g_error ("X Error: %s",  msg);
   exit (EXIT_FAILURE);
}

void x_init (XWindow *xwindow, guint width, guint height) 
{
   XSetErrorHandler (x_error);
   
   if ( ( xwindow->display = XOpenDisplay (NULL)) == NULL) 
     {
	g_error ("Can't open display");
	exit (EXIT_FAILURE);
     }
   
   XSynchronize (xwindow->display, True);
   
   xwindow->screen = DefaultScreen (xwindow->display);
   xwindow->depth  = DefaultDepth (xwindow->display, xwindow->screen);
   xwindow->root   = RootWindow (xwindow->display, xwindow->screen);
   Window root;
   XGetGeometry (xwindow->display, xwindow->root, &root,
		 &xwindow->x, &xwindow->y,
		 &xwindow->root_width, &xwindow->root_height,
		 &xwindow->border, &xwindow->depth);

   XSetWindowAttributes window_attributes;
   window_attributes.override_redirect = True;
   
   xwindow->window = XCreateWindow (xwindow->display, xwindow->root, 0, 0,
				    width, height, 0, xwindow->depth, InputOutput, CopyFromParent,
				    CWOverrideRedirect, &window_attributes);
   
   gdk_pixbuf_xlib_init (xwindow->display, xwindow->screen);
   
   XSelectInput (xwindow->display, xwindow->window,
		 ExposureMask | VisibilityChangeMask | KeyPressMask | ButtonPressMask | ButtonReleaseMask);
   
   XSelectInput (xwindow->display, xwindow->root,
		 ButtonPressMask | ButtonReleaseMask);
   
   XMapWindow (xwindow->display, xwindow->window);
   
   XFlush (xwindow->display);
}

void x_update_sprite ( Souris *souris)
{
   guint click_no = 0;
   
   if ( souris->left_button_pressed)
     click_no += 1;
   
   if ( souris->middle_button_pressed)
     click_no += 2;
   
   if ( souris->right_button_pressed)
     click_no += 4;
   
   if ( click_no != souris->current_sprite )
     {
	souris->current_sprite = click_no;
	sprite_paint (souris->xwindow, souris->clicks_sprites[souris->current_sprite]);
     }
}

void x_update_position ( Souris *souris) 
{

   Window root, child;
   gint absolute_x, absolute_y;
   gint relative_x, relative_y;
   guint mod_key_mask;
   
   // Get absolute (& relative) mouse coord.
   XQueryPointer (souris->xwindow->display, souris->xwindow->window, &root, &child,
		  &absolute_x, &absolute_y, &relative_x, &relative_y,
		  &mod_key_mask);
   
   XWindowChanges changes;
   changes.x = absolute_x - PIXMAP_WIDTH_PX/2;
   changes.y = absolute_y - PIXMAP_HEIGHT_PX;
   
   XConfigureWindow (souris->xwindow->display, souris->xwindow->window, CWX | CWY, &changes);
}

gboolean x_event (GIOChannel *source, GIOCondition condition, gpointer data) 
{  
   Souris *souris = (Souris *) data;
   
   XEvent event;
   
   while ( XPending (souris->xwindow->display)) 
     {

	XNextEvent (souris->xwindow->display, &event);
	
	switch (event.type) 
	  {
	   case Expose:
	     if (event.xexpose.count == 0)
	       on_expose (souris);
	     break;
	   case ButtonPress:			
	     break;
	   case ButtonRelease:
	     break;
	   case VisibilityNotify:
	     XRaiseWindow (souris->xwindow->display, souris->xwindow->window);
	     break;
	   default:
	     break;
	  }
     }
   
   x_update_position (souris);
   sprite_repaint (souris->xwindow, souris->clicks_sprites[souris->current_sprite]);

   return true;
}

gboolean x_timeout (gpointer data) 
{
   Souris *souris = (Souris *) data;
      
   x_update_position (souris);

   x_update_sprite (souris);
   
   //sprite_paint (souris->xwindow, souris->clicks_sprites[souris->current_sprite]);

   return true;
}

Souris * souris_init () 
{
   struct Souris *souris;
   
   souris = ( struct Souris * ) malloc ( sizeof (struct Souris) );

   souris->clicks_sprites = ( struct Sprite ** ) calloc ( 10, sizeof (struct Sprite *));
   souris->xwindow = ( struct XWindow * ) malloc ( sizeof ( struct XWindow) );
   souris->xwindow_thread = ( struct XWindow * ) malloc ( sizeof ( struct XWindow) );
 
   souris->left_button_pressed = false;
   souris->middle_button_pressed = false;
   souris->right_button_pressed = false;
   
   return souris;
}

void souris_post_init( Souris *souris)
{
   int s;
   char filename[256];
   
   for ( s = 0 ; s < 8 ; s++ )
     {
	sprintf ( filename, "%s/clicks_%d.png", SOURIS_DATA_DIR, s);
	souris->clicks_sprites[s] = sprite_init ( souris->xwindow, filename);
     }
   
   souris->current_sprite = 3;   
   
   souris->mutex = g_mutex_new();
}


Sprite * sprite_init (XWindow *xwindow, char *filename)
{
   Sprite *sprite; 
   GdkPixbuf *	pixbuf;
  
   pixbuf = gdk_pixbuf_new_from_file (filename, 0);
   
   if ( pixbuf == 0 )
     {
	fprintf (stderr, "Can't load pixmap %s.\n", filename);
	exit (EXIT_FAILURE);	
     }
     
   
   sprite = ( Sprite *) malloc (  sizeof ( struct Sprite) ); 
      
   sprite->width = gdk_pixbuf_get_width (pixbuf);
   sprite->height = gdk_pixbuf_get_height (pixbuf);
   
   XGCValues gc_values;
   gc_values.function = GXcopy;
   gc_values.fill_style = FillTiled;
   gc_values.ts_x_origin = 0;
   gc_values.ts_y_origin = 0;
   
   gdk_pixbuf_xlib_render_pixmap_and_mask (pixbuf, &sprite->pixmap, &sprite->mask, 127);
   
   gc_values.tile = sprite->pixmap;
   
   sprite->gc = XCreateGC (xwindow->display, xwindow->window,
			   GCFunction | GCTile | GCTileStipXOrigin | GCTileStipYOrigin | GCFillStyle,
			   &gc_values);
   
   g_object_unref (pixbuf);

   return sprite;
}

void sprite_paint (XWindow *xwindow, Sprite *sprite) 
{
   XShapeCombineMask (xwindow->display, xwindow->window,
		      ShapeBounding, 0, 0, sprite->mask, ShapeSet);
   
   sprite_repaint (xwindow, sprite);
}


void sprite_repaint (XWindow *xwindow, Sprite *sprite) 
{
   XFillRectangle (xwindow->display, xwindow->window, sprite->gc,
		   0, 0, sprite->width, sprite->height);
   XFlush (xwindow->display);
}

void on_expose ( Souris *souris) 
{
   XFillRectangle (souris->xwindow->display, souris->xwindow->window, souris->clicks_sprites[souris->current_sprite]->gc,
		   0, 0, 100, 60);
   XFlush (souris->xwindow->display);
}

static void on_record_event ( XPointer xpointer, XRecordInterceptData *data )
{
   if ( data == NULL )
     {
	fprintf (stderr,"on_record_event : NULL received\n");
	return;
     }
   
   if ( data->data == NULL || data->data_len < 1 )
     return;
   
   Souris *souris = (Souris *)xpointer;
   
   XRecordDatum * d = (XRecordDatum *) data->data;
   
   if ( d->type == ButtonPress )
     {
	xEvent xe = d->event;

	// Je ne trouve pas suffisament de doc sur les meandres de la Xlib
	// essayons avec la methode empirique ;)  
	
	unsigned char b = xe.u.u.detail;
	
//	fprintf( stderr, "Button %d pressed\n",b);

	if ( b == 1 )
	  souris->left_button_pressed = true;
	else if ( b == 2 )
	  souris->middle_button_pressed = true;
	else if ( b == 3 )
	  souris->right_button_pressed = true;     
     }
   else if ( d->type == ButtonRelease )
     {
	xEvent xe = d->event;

	unsigned char b = xe.u.u.detail;
	
//	fprintf( stderr, "Button %d pressed\n",b);

	if ( b == 1 )
	  souris->left_button_pressed = false;
	else if ( b == 2 )
	  souris->middle_button_pressed = false;
	else if ( b == 3 )
	  souris->right_button_pressed = false;     
     }
   else if ( d->type == MotionNotify )
     {
//	g_mutex_lock (souris->mutex);		
//	g_mutex_unlock (souris->mutex);
     }
      
   XRecordFreeData(data);
}

void * events_watcher( void * data )
{
   Souris *souris = (Souris *) data;
   
   Display* display = XOpenDisplay(0);
   
   if ( display == 0 ) 
     {
	fprintf(stderr, "events_watcher_thread: unable to open display\n");
	return NULL;
     }

   XRecordClientSpec xrecord_client_spec = XRecordAllClients; 
   
   XRecordRange **xrecord_ranges = (XRecordRange **)malloc ( 5 * sizeof( XRecordRange *) );
   xrecord_ranges[0] = XRecordAllocRange();
   
   xrecord_ranges[0]->device_events.first = KeyPress;
   xrecord_ranges[0]->device_events.last = MotionNotify;
   
   XRecordContext xr_context = XRecordCreateContext( display,
						     XRecordFromServerTime|XRecordFromClientTime|XRecordFromClientSequence,
						     &xrecord_client_spec,
						     1,
						     xrecord_ranges,
						     1);   
   
   XRecordEnableContext( display, xr_context, on_record_event, (char *)souris);

   return NULL;
}
   

int main ( int argc, char **argv ) 
{
   // init
   Souris *souris = souris_init();   

   // GTK init
   gtk_init (&argc, &argv);

   g_thread_init (NULL);

   // X init of souris window
   x_init (souris->xwindow, 100, 60);
   
   // Initialize souris (can only be done once xwindow has been initialized)
   souris_post_init (souris);
   
   x_update_position (souris);
   
   // Plug a watch on x connection socket into gtk main
   GIOChannel *channel = g_io_channel_unix_new (ConnectionNumber(souris->xwindow->display));
   
   g_io_add_watch_full (channel, G_PRIORITY_HIGH,
			(GIOCondition)(G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL),
			(GIOFunc) x_event, souris, 0);
   
   // Paint window
   sprite_paint (souris->xwindow, souris->clicks_sprites[souris->current_sprite]);
   
   // Timeout
   int timetag = g_timeout_add ( 10, x_timeout, souris);
      
   int ext_record_major;
   int ext_record_minor;
   
   if ( XRecordQueryVersion( souris->xwindow->display,
			     &ext_record_major,
			     &ext_record_minor     ) == 0 )
     {
	
	fprintf (stderr, "X Record extension is missing.\n");;
     }
   else
     {
	fprintf (stdout, "X Record extension %d.%d is loaded",ext_record_major,ext_record_minor);
     }


   GThread* thread = g_thread_create ( events_watcher, souris, TRUE, NULL);
   
   if ( thread == NULL )
     {
	fprintf (stderr, "thread error\n");
	exit (EXIT_FAILURE);
     }
   
   gtk_main();
   
   // Remove timeout
   g_source_remove (timetag);
   
   exit (EXIT_SUCCESS);
}

