#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <memory.h>
#include "VSDL_sprite.h"

VSDL_fonts* vsdl_currentFonts ;

extern int vsdl_fontsReady ;
extern SDL_Surface *vsdl_screen ;
extern int vsdl_videoReady ;
extern char vsdl_path[] ;

/* Load an image file in a pixmap */
VSDL_pixmap* VSDL_LoadPixmap( char* filename, char tr, char alpha, int flags) {
   VSDL_pixmap* new_pixmap = NULL ;
   SDL_Surface* new_surface = NULL ;
   char filepath[512] ;
   
   if ( vsdl_path != NULL )
     sprintf( filepath, "%s/%s", vsdl_path, filename) ;
   else
     strncpy( filepath, filename, 512) ;
   new_surface = IMG_Load( filepath) ;
   if ( new_surface == NULL ) {
      fprintf( stderr, "VSDL: Couldn't load %s: %s\n",
	      filepath,SDL_GetError());
      return (NULL) ;
   }
   new_pixmap = new VSDL_pixmap ;
   strncpy( new_pixmap->name, filename, 256) ;
   new_pixmap->transparency = tr ;
   new_pixmap->alpha = alpha ;
   if ( flags == 0 )
     new_pixmap->image = SDL_DisplayFormat( new_surface) ;
   else 
     new_pixmap->image = SDL_ConvertSurface( new_surface, vsdl_screen->format, SDL_SWSURFACE) ;
   if ( new_pixmap->image == NULL ) {
      delete new_pixmap ;
      return( NULL) ;
   } 
   SDL_FreeSurface( new_surface) ;
   if ( new_pixmap->transparency == 1 ) { /* Set the transparency */
      SDL_SetColorKey( new_pixmap->image, SDL_SRCCOLORKEY,
		      SDL_MapRGB( new_pixmap->image->format, 0, 0, 0)) ;
   }
   if ( new_pixmap->alpha != 0 ) { /* Set the alpha */
      SDL_SetAlpha( new_pixmap->image, SDL_SRCALPHA, new_pixmap->alpha) ;
   }
   new_pixmap->nbFrames = 0 ;
   new_pixmap->speed = 0 ;
   new_pixmap->currentTime = 0 ;
   new_pixmap->currentFrame = 0 ;
   new_pixmap->ready = 1 ;
   return( new_pixmap) ;
}

/* Put a pixmap at x, y */
int VSDL_PutPixmap(VSDL_pixmap* pixmap, int x, int y) {
   SDL_Rect dest ;
   SDL_Rect src ;
   
   if ( pixmap == NULL )
     return( -1) ;
   if ( pixmap->ready != 1 ) 
     return( 0) ;
   
   if ( pixmap->nbFrames == 0 ) {
      dest.x = x ;
      dest.y = y ;
      dest.w = pixmap->image->w ;
      dest.h = pixmap->image->h ;
      SDL_BlitSurface( pixmap->image, NULL, vsdl_screen, &dest) ; 
   }
   else {
      dest.x = x ;
      dest.y = y ;
      dest.w = pixmap->image->w ;
      dest.h = pixmap->image->h / pixmap->nbFrames ;
      src.x = 0 ;
      src.y = dest.h * pixmap->currentFrame ;
      src.w = dest.w ;
      src.h = dest.h ;
      SDL_BlitSurface( pixmap->image, &src, vsdl_screen, &dest) ;
   }
   return( 1) ;
}

/* Scale a pixamp */
VSDL_pixmap* VSDL_ScalePixmap( VSDL_pixmap* pixmap, int nw,int nh) {
   VSDL_pixmap* npix ;
   int ow, oh ;
   int x, y, h ;
   
   if ( pixmap == NULL )
     return( NULL) ;
   if ( pixmap->ready != 1 )
     return( NULL) ;
     
   npix = new VSDL_pixmap ;
   npix->image = SDL_AllocSurface( SDL_SWSURFACE, nw, nh,
				  pixmap->image->format->BitsPerPixel,
				  pixmap->image->format->Rmask,
				  pixmap->image->format->Gmask,
				  pixmap->image->format->Bmask,
				  pixmap->image->format->Amask ) ;
   
   ow = pixmap->image->w ;
   oh = pixmap->image->h ;
   if ( SDL_MUSTLOCK( pixmap->image))
     SDL_LockSurface( pixmap->image) ;
   if ( pixmap->image->format->BitsPerPixel == 16 ) {
      for(y=0;y<nh;y++) {
	 h = y*oh/nh ;
	 for(x=0;x<nw;x++)
	   *((Uint16*)npix->image->pixels+y*nw+x) = *((Uint16*)pixmap->image->pixels+h*ow+x*ow/nw) ;
      }
   } 
   else if ( pixmap->image->format->BitsPerPixel == 32 ) {
      for(y=0;y<nh;y++) {
	 h = y*oh/nh ;
	 for(x=0;x<nw;x++)
	   *((Uint32*)npix->image->pixels+y*nw+x) = *((Uint32*)pixmap->image->pixels+h*ow+x*ow/nw) ; 
      }  
   }
   if ( SDL_MUSTLOCK( pixmap->image))
     SDL_UnlockSurface( pixmap->image) ;
   npix->transparency = pixmap->transparency ;
   npix->alpha = pixmap->alpha ;
   if ( npix->transparency == 1 ) { /* Set the transparency */
      SDL_SetColorKey( npix->image, SDL_SRCCOLORKEY,
		      SDL_MapRGB( npix->image->format, 0, 0, 0)) ;
   }
   if ( npix->alpha != 0 ) { /* Set the alpha */
      SDL_SetAlpha( npix->image, SDL_SRCALPHA, npix->alpha ) ;
   }
   npix->nbFrames = 0 ;
   npix->speed = 0 ;
   npix->currentTime = 0 ;
   npix->currentFrame = 0 ;
   npix->ready = pixmap->ready ;
   return( npix) ;
}   

/* Rotate a pixmap */
VSDL_pixmap* VSDL_RotatePixmap( VSDL_pixmap* pixmap, float angle) {
   VSDL_pixmap* npix ;
   float sinAngle, cosAngle ;
   int w, h, wd2, hd2 ;
   int i, j ;
   int x1, y1, x2,y2 ;
   
   if ( pixmap == NULL)
     return( NULL) ;
  
   w = pixmap->image->w ;
   h = pixmap->image->h ;
   wd2 = w >> 1 ;
   hd2 = h >> 1 ;
   npix = new VSDL_pixmap ;
   npix->image = SDL_AllocSurface( SDL_SWSURFACE, w, h,
				  pixmap->image->format->BitsPerPixel,
				  pixmap->image->format->Rmask,
				  pixmap->image->format->Gmask,
				  pixmap->image->format->Bmask,
				  pixmap->image->format->Amask ) ;
   sinAngle = sin( angle) ;
   cosAngle = cos( angle) ;
   if ( SDL_MUSTLOCK( pixmap->image))
     SDL_LockSurface( pixmap->image) ;
   if ( pixmap->image->format->BitsPerPixel == 16 ) {
      for(j=0;j<h;j++) { 
	 y1 = j - hd2 ;
	 for(i=0;i<w;i++) {
	    x1 = i - wd2 ;
	    x2 = (int) ( x1*cosAngle - y1*sinAngle ) + wd2 ;
	    y2 = (int) ( x1*sinAngle + y1*cosAngle ) + hd2 ;
	    if ( x2 > -1 && x2 < w && y2 > -1 && y2 < h )
	      *((Uint16*)npix->image->pixels+w*j+i) = *((Uint16*)pixmap->image->pixels+w*y2+x2) ;
	    else
	      *((Uint16*)npix->image->pixels+w*j+i) = 0 ;
	 }
      }
   }
   else if ( pixmap->image->format->BitsPerPixel  == 32 ) {
      for(j=0;j<h;j++) {
	 y1 = j - ( h >> 2 ) ;
	 for(i=0;i<w;i++) {
	    x1 = i - ( w >> 2 ) ;
	    x2 = (int) ( x1*cosAngle - y1*sinAngle ) + ( w >> 2 ) ;
	    y2 = (int) ( x1*sinAngle + y1*cosAngle ) + ( h >> 2 ) ;
	    if ( x2 > -1 && x2 < w && y2 > -1 && y2 < h )
	      *((Uint32*)npix->image->pixels+w*j+i) = *((Uint32*)pixmap->image->pixels+w*y2+x2) ;      
	    else 
	      *((Uint32*)npix->image->pixels+w*j+i) = 0 ;
	 }
      }
   }
   if ( SDL_MUSTLOCK( pixmap->image))
     SDL_UnlockSurface( pixmap->image) ;
   npix->transparency = pixmap->transparency ;
   npix->alpha = pixmap->alpha ;
   if ( npix->transparency == 1 ) { /* Set the transparency */
      SDL_SetColorKey( npix->image, SDL_SRCCOLORKEY, 
   		      SDL_MapRGB( npix->image->format, 0, 0, 0)) ;
   }
   if ( npix->alpha != 0 ) { /* Set the alpha */
      SDL_SetAlpha( npix->image, SDL_SRCALPHA, npix->alpha ) ;
   }
   npix->nbFrames = 0 ;
   npix->speed = 0 ;
   npix->currentTime = 0 ;
   npix->currentFrame = 0 ;
   npix->ready = pixmap->ready ;
   return( npix) ;
}

/* free a VSDL_pixmap */
void VSDL_FreePixmap( VSDL_pixmap* pixmap) {

   if ( pixmap == NULL)
     return ;
   pixmap->ready = 0 ;
   SDL_FreeSurface( pixmap->image) ;
   delete pixmap ;
}

int VSDL_InitAnimatedPixmap( VSDL_pixmap* pixmap, char nbFrames, char speed) {
   
   pixmap->nbFrames = nbFrames ;
   pixmap->speed = speed ;
   return( 1) ;
}

int VSDL_PushAnimatedPixmap( VSDL_pixmap* pixmap) {
   
   pixmap->currentTime-- ;
   if ( pixmap->currentTime < pixmap->speed ) {
      pixmap->currentTime = 255 ;
      pixmap->currentFrame++ ;
      if ( pixmap->currentFrame >= pixmap->nbFrames )
	pixmap->currentFrame = 0 ;
   }
   return( 1) ;
}

void VSDL_Print( int x, int y, VSDL_pixmap* font, char* text) {
   int lng, i ;
   int fw, fh ;
   SDL_Rect rsrc, rdst ;
   
   fw = font->image->w / 10 ;
   fh = font->image->h / 10 ;
   rsrc.w = fw ;
   rsrc.h = fh ;
   rdst.w = fw ;
   rdst.h = fh ;
   lng = strlen( text) ;
   for(i=0;i<lng;i++) {
      rsrc.x = fw * ((text[i]-33) % 10) ;
      rsrc.y = fh * ((text[i]-33) / 10) ;
      rdst.x = x + fw * i ;
      rdst.y = y ;
      SDL_BlitSurface( font->image, &rsrc, vsdl_screen, &rdst) ;
   }
}
