#include "midi_sequencer.hpp"
#include "midi_timer.hpp"
#include "midi_device.hpp"
#include "midi_file.hpp"
#include "midi_event.hpp"
#include "midi_lyric.hpp"
#include "display_manager.hpp"
#include <stdlib.h>
#include <SDL.h>
#include <SDL_timer.h>

#define MAX_INT                10000000
#define DEFAULT_TEMPO          120
#define USEC_IN_1_MIN          60000000

Midi_Sequencer::Midi_Sequencer()
{
   file = NULL;
   device = NULL;
   timer = NULL;
   display_manager = NULL;
   lyrics = NULL;
   state = UNINITIALIZED;
}

Midi_Sequencer::~Midi_Sequencer()
{
   if ( file)
     delete file;
   if ( lyrics)
     delete lyrics;
   if ( device)
     delete device;
   if ( display_manager)
     delete display_manager;
   if ( timer)
     delete timer;
}

int Midi_Sequencer::Initialize( Midi_Device* device)
{
   this->device = device;
   device->Initialize();
   timer = new Midi_Timer();
   timer->Initialize();
   display_manager = new DisplayManager();
   display_manager->Initialize( "./");
   return 0;
}

int Midi_Sequencer::SetDisplay( char *display_name)
{
   display_manager->LoadDisplayPlugin( display_name);
   return 0;
}

int Midi_Sequencer::LoadFromFile( char* filename)
{
   file = new Midi_File();
   file->Load( filename);
   if ( lyrics) 
     delete lyrics;
   lyrics = new Midi_Lyric( file); 
   display_manager->SetLyric( lyrics);
   return 0;
}

void Midi_Sequencer::Unload()
{
   if ( file)
     delete file;
   file = NULL;
}

void Midi_Sequencer::Play()
{
   // TODO: test state
   
   state = PLAY;
   
   Midi_File *mf = file;
   int *current_pos_in_tracks = new int[mf->GetNbTracks()];
   
   for(unsigned int t=0;t<mf->GetNbTracks();t++)
     current_pos_in_tracks[t] = 0 ;
     
   int next_event_minus_time;
   int next_event_is_from_track; 
    
   timer->SetMidiTickDuration( (unsigned int) ( (double)USEC_IN_1_MIN / (double)DEFAULT_TEMPO / (double)file->division));
   timer->Start();
   
   while( state == PLAY) 
     {		
	// on cherche le prochain event dans tous les tracks
       	// 
	next_event_minus_time = MAX_INT;
	next_event_is_from_track = -1;
	
	for(unsigned int t=0;t<mf->GetNbTracks();t++) 
	  {
	     if ( current_pos_in_tracks[t] >= (int)mf->tracks[t].size()) // si ce track est termine on passe au suivant
	       continue;
	     
	     if ( (int)mf->tracks[t][current_pos_in_tracks[t]]->absolute_time < next_event_minus_time) 
	       // on  en a trouve un plus proche
	       {
		  next_event_minus_time = mf->tracks[t][current_pos_in_tracks[t]]->absolute_time; 
		  next_event_is_from_track = t;
	       }
	  }
	
	// s'il n'y en a pas, la chanson est terminee...
	if ( next_event_is_from_track == -1)
	  {
	     state = STOP;
	     fprintf( stderr, "C'EST FINI!\n");
	  }
	else
	  {
	     // si on l'a on le traite 
	     ProcessEvent( mf->tracks[next_event_is_from_track][current_pos_in_tracks[next_event_is_from_track]]);  

	     if ( display_manager != NULL)
	       display_manager->Update( timer->GetMidiTicks());
	     
	     // on avance dans le track
	     current_pos_in_tracks[next_event_is_from_track] += 1;
	     
	     //fprintf( stdout, "\r current_time=%d", current_time);
	  }  
     }
}

void Midi_Sequencer::ProcessEvent( Midi_Event* event)
{

   timer->WaitUntilMidiTick( event->absolute_time); 
   
   device->SendEvent( event);
   
   if ( event->type == Midi_Event::META)
     {
	if ( event->metaEventCode == MIDI_SET_TEMPO)
	  {
	     unsigned int noire_duration = // duree d'une noire en usec 
	       ((((unsigned char*)event->metaMessage)[0])<<16)&0x00FF0000 |
	       ((((unsigned char*)event->metaMessage)[1])<<8)&0x0000FF00  |
	       ((unsigned char*)event->metaMessage)[2] & 0x000000FF ; 

	     double current_tempo = USEC_IN_1_MIN / noire_duration;
	     unsigned int tick_duration = (unsigned int) ( (double)noire_duration / (double)file->division); // TODO: erreur d'approximation 
	     
	     // 1 division <=> tmpo usec
	     // 1 beat <=> 4*nb_divisons
	     // so 1 beat last 4*nbdivisions**tmpo usec
	      
	     timer->SetMidiTickDuration( tick_duration);
	     fprintf( stderr, "SET TEMPO to %f | tick duration=%d\n", current_tempo, tick_duration);
	  }
	else if ( event->metaEventCode == MIDI_SMPTE_OFFSET )
	  {
	     fprintf( stderr, "SMPTE: DESOLE je n'ai pas encore code cela (je suis avec Emilie et la on fait du karaoke alors pas le temps de coder ca!!!!\n");
	  }
	else if ( event->metaEventCode == MIDI_TIME_SIGNATURE)
	  {
	     fprintf( stderr, "METRONOME:\n");
	     short time_signature_numerator = event->metaMessage[0];
	     short time_signature_denominator = event->metaMessage[1];
	     short midi_clocks_per_metronome_ticks = event->metaMessage[2];
	     short nb_1_32_note_per_miid_clock = event->metaMessage[3];
	     
	     fprintf( stderr, "-> time_signature_numerator=%d\n", time_signature_numerator);
	     fprintf( stderr, "-> time_signature_denominator=%d\n", time_signature_denominator);
	     fprintf( stderr, "-> midi_clocks_per_metronome_ticks=%d\n", midi_clocks_per_metronome_ticks);
	     fprintf( stderr, "-> nb_1_32_note_per_miid_clock=%d\n", nb_1_32_note_per_miid_clock);
	  }
     }   
}

// TODO: a single Action( actiontype)?
// 
//void Midi_Sequencer::Play();
//void Midi_Sequencer::Pause();
//void Midi_Sequencer::Stop();
