#include <string>
#include <SDL.h>
#include <SDL_endian.h>
#include "midi_file.hpp"
#include "midi_event.hpp"

#define MIDI_FILE_MTHD      "MThd"
#define MIDI_FILE_MTRK      "MTrk"

struct Chunk_Header
{
   typedef enum _type { 
      MTHD, 
      MTRK, 
      UNKNOW
   };
   _type type; 
   unsigned int length;
};

Midi_File::Midi_File()
{
   f = NULL;
}

Midi_File::~Midi_File()
{ 
}

void Midi_File::Load( char *filename)
{
   fprintf( stdout, "Midi_File::Load: Loading file %s\n", filename);
   
   f = SDL_RWFromFile( filename, "r");
   if ( f == NULL) {
      fprintf( stderr, "Midi_file::Load: Can't open file %s!\n", filename);
      return;
   }
 
   // 1) search the Midi Header (MThd)
   
   bool header_found = false;
   
   while( !header_found) // TODO et tester fin de fichier
     {
	Chunk_Header *header = ReadChunkHeader();
	
	if ( header->type == Chunk_Header::MTHD) // Ok, we have a MThd
	  {
	     format = ReadShort();
	     
	     nb_tracks = ReadShort();
	     
	     if ( format == 0 && nb_tracks != 1 )
	       fprintf( stderr, "Midi_File::Load: midi format %d and nb_tracks=%d (standard say that nb_tracks must be 1)\n", format, nb_tracks);
	     
	     if ( format == 2 )
	       fprintf( stderr, "TODO: format=%d not supported\n", format);
	     
	     division = ReadShort();
	     
	     if ( ! ( division &  0x8000) )
	       {
		  // nb division in a quarter of note
		  fprintf( stdout, "divisons in 1/4 note=%d\n", division);
	       }
	     else 
	       {
		  // SMPTE time
		  fprintf( stderr, "TODO: file use SMPTE time for division\n"); 	       

		  short frames_per_second = - ( ( division >> 8 ) & 0x00FF);
		  
		  short ticks_per_frames = division & 0x00FF;
		  
		  fprintf( stderr, "SMPTE found -> fps=%d tpf=%d\n", frames_per_second, ticks_per_frames);
	       }
	     
	     header_found = true;
	  }
	else // not a Mthd, skip this chunk
	  {
	     fprintf( stderr, "Midi_File::Load: skip unknown header\n");
	     SDL_RWseek( f, header->length, SEEK_CUR);
	  }	
     }
      
   // 2) Now we have collected needed infos, go reading tracks
	   
   for( int t=0;t<nb_tracks;t++)
     {
	tracks.push_back( vector_of_midi_events());
	header_found = false;
	
	// Search a Track Header (MTrk)
	
	while( !header_found)
	  {
	     Chunk_Header *header = ReadChunkHeader();
	     
	     if ( header->type == Chunk_Header::MTRK) // Ok, we have a MTrk
	       {
		  // So fill the corresponding track
	
		  //fprintf( stdout, "MTrk chunk found for track %u (length %u)\n", t, header->length);
	     
		  unsigned int track_begining = SDL_RWtell( f);
		  
		  last_time = 0; // so as ReadEvent() can compute absolute_time from delta_time
		  
		  while( (unsigned int)SDL_RWtell( f) < track_begining+header->length)
		    {
		       Midi_Event *event = ReadEvent();
		       if ( event)
			 tracks[t].push_back( event);
		    }
		  
		  // If track is not of expected size, repositionning
		  if ( (unsigned int)SDL_RWtell( f) != track_begining+header->length)
		    {
		       fprintf( stderr, "Midi_File::Load: end of track at %d while expected at %d\n",SDL_RWtell( f),track_begining+header->length);
		       SDL_RWseek( f, track_begining+header->length, SEEK_SET);
		    }
	    
		  header_found = true;
	       }
	     else // not a MTrk, skip this chunk
	       {
		  fprintf( stderr, "Midi_File::Load: skip unknown header\n");
		  SDL_RWseek( f, header->length, SEEK_CUR);
	       }	
	  }	
     }
   
   SDL_RWclose( f);
   
   fprintf( stdout, "Midi_File::Load: %s successfully loaded\n", filename); 
}

Chunk_Header *Midi_File::ReadChunkHeader()
{
   Chunk_Header *header = new Chunk_Header();
   char header_string[4];
   
   SDL_RWread( f, &header_string, sizeof(char), 4);
   
   header->length = ReadLong();
   
   if ( strncmp( header_string, MIDI_FILE_MTHD, 4) == 0 && header->length != 6)
     fprintf( stderr, "Midi_file::ReadChunkHeader: Found MThd of %d length\n", header->length);
	
   if ( strncmp( header_string, MIDI_FILE_MTHD, 4) == 0 && header->length == 6)
     header->type = Chunk_Header::MTHD;
   else if ( strncmp( header_string, MIDI_FILE_MTRK, 4) == 0)
     header->type = Chunk_Header::MTRK;
   else
     header->type = Chunk_Header::UNKNOW;
   
   return  header;
}

unsigned long Midi_File::ReadLong()
{
   return SDL_ReadBE32( f);
 }

unsigned short Midi_File::ReadShort()
{
   return SDL_ReadBE16( f);
}

unsigned char Midi_File::ReadByte()
{
   unsigned char byte=0;
   
   if ( SDL_RWread( f, &byte, sizeof(char), 1) != 1 )
     fprintf( stderr, "ReadByte error\n");
   return byte;
}

unsigned long Midi_File::ReadVariableLength()
{
   unsigned long vl;
   unsigned char byte; 
   
   byte = ReadByte(); 
   vl = byte;
   
   if ( byte & 0x80)
     {  
	vl &= 0x7F;
	do {
	   byte = ReadByte();
	   vl = ( vl << 7) + ( byte & 0x7F );
	   
	} while( byte & 0x80);
     }
   
   return vl;
}

Midi_Event *Midi_File::ReadEvent()
{
   static unsigned int eventCode = 0x80; //0x08
   unsigned char byte;
   
   Midi_Event *event = new Midi_Event();
   
   event->delta_time = ReadVariableLength();
   last_time += event->delta_time;   
   event->absolute_time = last_time;
   
   // Read the first byte
   byte = ReadByte();
   if ( byte & MIDI_STATUS_BYTE_MASK) // it's a running status change
     {
	eventCode = byte;
     }
   else // this byte is a piece of event, it will be readen later
     {
	//fprintf( stderr, "Midi_File::Read_Event: not a runnning status change -> seek -1\n");
        SDL_RWseek( f, -1, SEEK_CUR);
     }
      
   event->eventCode = eventCode;
   
   if ( eventCode == MIDI_FILE_META_EVENT) // this is a META event
     { 
	event->type = Midi_Event::META;
	event->metaEventCode = ReadByte();
	event->messageLength = ReadVariableLength();
	event->metaMessage = new char[event->messageLength];
	SDL_RWread( f, event->metaMessage, sizeof(char), event->messageLength); // TODO: add a method to do this like ReadByte()...	

	/*
	if (metaEventCode == MIDI_TIME_SIGNATURE ||
	    metaEventCode == MIDI_SET_TEMPO) {
	   m_containsTimeChanges = true;
	}
	
	if (defaultChannel >= 0) {
	   long gap = accumulatedTime - channelTimeMap[defaultChannel];
	   accumulatedTime += deltaTime;
	   deltaTime += gap;
	   channelTimeMap[defaultChannel] = accumulatedTime;
	   m_midiComposition[channelTrackMap[defaultChannel]].push_back(e);
	} else {
	   m_midiComposition[trackNum].push_back(e);
	}
        */	
     }
   else // MIDI or SYSEX
     { 
	int channel = eventCode & MIDI_CHANNEL_NUM_MASK;
	/*if (channelTrackMap[channel] == -1) {
	   if (!firstTrack) ++trackNum;
	   else {
	      defaultChannel = channel;
	      firstTrack = false;
	   }
	   channelTrackMap[channel] = trackNum;
	}
	  
	  long gap = accumulatedTime - channelTimeMap[channel];
	 accumulatedTime += deltaTime;
	    deltaTime += gap;
	    channelTimeMap[channel] = accumulatedTime;
*/
	event->type = Midi_Event::MIDI;
	
	switch( eventCode & MIDI_MESSAGE_TYPE_MASK)
	  {
	   case MIDI_NOTE_ON:
	   case MIDI_NOTE_OFF:
	   case MIDI_POLY_AFTERTOUCH:
	   case MIDI_CTRL_CHANGE:
	   case MIDI_PITCH_BEND:
	     
	     event->data1 = ReadByte();
	     event->data2 = ReadByte();
	     break;
	     
	   case MIDI_PROG_CHANGE:
	   case MIDI_CHNL_AFTERTOUCH:
	     
	     event->data1 = ReadByte();
	     break;
	     
	   case MIDI_SYSTEM_EXCLUSIVE:
	     event->type = Midi_Event::SYSEX;
	     event->messageLength = ReadVariableLength();
	     event->metaMessage = new char[event->messageLength];
	     SDL_RWread( f, event->metaMessage, sizeof(char), event->messageLength);
	     if ( ((unsigned char*)event->metaMessage)[event->messageLength-1] != MIDI_END_OF_EXCLUSIVE)
	       {
		  //TODO
		  fprintf( stderr, "Midi_File::ReadEvent: malformed or unsupported SysEx event!\n");
		  delete event;
		  return NULL;
	       }
	     ReadByte();	     
	     // TODO: sortir ca mieux le End Of exclusivemetaMessage -> -2
	     event->messageLength -= 1;
	     break;
	     
	   case MIDI_END_OF_EXCLUSIVE:
	     event->type = Midi_Event::SYSEX;
	     // TODO: quoi????
	     fprintf( stderr, "Midi_File::ReadEvent: found a stray MIDI_END_OF_EXCLUSIVE");
	     delete event;
	     return NULL; 
	     break;
	     
	   default:
	     fprintf( stderr, "Midi_File::Read_Event: unsupported Midi event code: %d\n", eventCode); 
	     delete event;
	     return NULL;
	     break;
	  } 
     }
   
   //event->print();
   return event;
}
