#include "midi_device_alsa.hpp"
#include <alsa/asoundlib.h>
#include <alsa/seq_event.h>
#include <alsa/version.h>
#include "midi_event.hpp"

Midi_Device_Alsa::Midi_Device_Alsa()
{  
}

Midi_Device_Alsa::~Midi_Device_Alsa()
{
   snd_seq_stop_queue( handle, queue, 0);
   snd_seq_close( handle);
}

/*
void Midi_Device_Alsa::getSystemInfo()
{
    int err;
   snd_seq_system_info_t *sysinfo;
   
   snd_seq_system_info_alloca(&sysinfo);
   
   if ((err = snd_seq_system_info(m_midiHandle, sysinfo)) < 0)
     {
        std::cerr << "System info error: " <<  snd_strerror(err)
	  << std::endl;
        exit(EXIT_FAILURE);
     }
   
   m_maxQueues = snd_seq_system_info_get_queues(sysinfo); 
   m_maxClients = snd_seq_system_info_get_clients(sysinfo);
   m_maxPorts = snd_seq_system_info_get_ports(sysinfo);
}

void AlsaDriver::showQueueStatus(int queue)
{
    int err, idx, min, max;
    snd_seq_queue_status_t *status;

    snd_seq_queue_status_alloca(&status);
    min = queue < 0 ? 0 : queue;
    max = queue < 0 ? m_maxQueues : queue + 1;

    for (idx = min; idx < max; ++idx)
    {
        if ((err = snd_seq_get_queue_status(m_midiHandle, idx, status))<0)
        {
            if (err == -ENOENT)
                continue;

            std::cerr << "Client " << idx << " info error: "
                      << snd_strerror(err) << std::endl;
            exit(EXIT_FAILURE);
        }

        std::cout << "Queue " << snd_seq_queue_status_get_queue(status)
                  << std::endl;

        std::cout << "Tick       = "
                  << snd_seq_queue_status_get_tick_time(status)
                  << std::endl;

        std::cout << "Realtime   = "
                  << snd_seq_queue_status_get_real_time(status)->tv_sec
                  << "."
                  << snd_seq_queue_status_get_real_time(status)->tv_nsec
                  << std::endl;

        std::cout << "Flags      = 0x"
                  << snd_seq_queue_status_get_status(status)
                  << std::endl;
    }

}


// Use this to hold all client information so that we can sort it
// before generating devices - we want to put non-duplex devices
// at the front of any device list (makes thing much easier at the
// GUI and we already have some backwards compatability issues with
// this).
//
class AlsaPortDescription
{
public:
    AlsaPortDescription(Instrument::InstrumentType type,
                        const std::string &name,
                        int client,
                        int port,
                        bool duplex):
        m_type(type),
        m_name(name),
        m_client(client),
        m_port(port),
        m_duplex(duplex) {;}

    Instrument::InstrumentType m_type;
    std::string                m_name;
    int                        m_client;
    int                        m_port;
    bool                       m_duplex;
};

// Sort by checking duplex state
//
struct AlsaPortCmp
{
    bool operator()(AlsaPortDescription *a1,
                    AlsaPortDescription *a2)
    {
        // Handle non-system clients by pushing them to the end of the
        // device list always.  This will keep devices in the order:
        //
        // o write-only system devices (soundcard synths)
        // o duplex system devices (MIDI ports)
        // o software devices (softsynths)
        //
        // I don't want to use the 128 here but this is at least
        // emperically correct for the moment and can't see a 
        // working alternative.
        //
        if (a1->m_client < 128 && a1->m_duplex != a2->m_duplex)
            return true;
        else
        {
            if (a1->m_client != a2->m_client)
                return a1->m_client < a2->m_client;
            else
                return a1->m_port < a2->m_port;
        }
    }
};

void
AlsaDriver::generateInstruments()
{
    snd_seq_client_info_t *cinfo;
    snd_seq_port_info_t *pinfo;
    int  client;
    unsigned int cap;

    snd_seq_client_info_alloca(&cinfo);
    snd_seq_client_info_set_client(cinfo, -1);

    // Reset these before each Device/Instrument hunt
    //
    m_deviceRunningId = 0;
    m_addedMetronome = false;
    m_audioRunningId = Rosegarden::AudioInstrumentBase;
    m_midiRunningId = Rosegarden::MidiInstrumentBase;
    m_currentPair  = std::pair<int, int>(-1, -1);

    // Clear these
    //
    m_instruments.clear();
    m_devices.clear();
    m_alsaPorts.clear();

    std::cout << std::endl << "  ALSA Client information:"
              << std::endl << std::endl;

    std::vector<AlsaPortDescription*> alsaPorts;

    // Get only the client ports we're interested in and store them
    // for sorting and then device creation.
    //
    while (snd_seq_query_next_client(m_midiHandle, cinfo) >= 0)
    {

        client = snd_seq_client_info_get_client(cinfo);
        snd_seq_port_info_alloca(&pinfo);
        snd_seq_port_info_set_client(pinfo, client);
        snd_seq_port_info_set_port(pinfo, -1);

        // ignore ourselves
        if (m_client == client) continue;

        while (snd_seq_query_next_port(m_midiHandle, pinfo) >= 0)
        {
            cap = (SND_SEQ_PORT_CAP_SUBS_WRITE|SND_SEQ_PORT_CAP_WRITE);

            if ((snd_seq_port_info_get_capability(pinfo) & cap) == cap)
            {
                std::cout << "    "
                          << snd_seq_port_info_get_client(pinfo) << ","
                          << snd_seq_port_info_get_port(pinfo) << " - ("
                          << snd_seq_client_info_get_name(cinfo) << ", "
                          << snd_seq_port_info_get_name(pinfo) << ")";

                if (snd_seq_port_info_get_capability(pinfo) &
                    SND_SEQ_PORT_CAP_DUPLEX)
                    std::cout << "\t\t\t(DUPLEX)";
                else
                    std::cout << "\t\t(WRITE ONLY)";

                // Generate a unique name using the client id
                //
                char clientId[10];
                sprintf(clientId,
                        "%d ",
                        snd_seq_port_info_get_client(pinfo));

                std::string fullClientName = 
                    std::string(snd_seq_client_info_get_name(cinfo));

                std::string clientName =
                    std::string(clientId) + fullClientName;

                AlsaPortDescription *portDescription = 
                    new AlsaPortDescription(
                            Instrument::Midi,
                            clientName,
                            snd_seq_port_info_get_client(pinfo),
                            snd_seq_port_info_get_port(pinfo),
                            snd_seq_port_info_get_capability(pinfo) &
                            SND_SEQ_PORT_CAP_DUPLEX);

                alsaPorts.push_back(portDescription);

                std::cout << std::endl;
            }
        }
    }
    std::cout << std::endl;

    // Ok now sort by duplexicity
    //
    std::sort(alsaPorts.begin(), alsaPorts.end(), AlsaPortCmp());

    std::vector<AlsaPortDescription*>::iterator it = alsaPorts.begin();
    for (; it != alsaPorts.end(); it++)
    {
        
        //cout << "installing device " << (*it)->m_name
        //     << " client = " << (*it)->m_client
        //     << " port = " << (*it)->m_port << endl;


        addInstrumentsForPort((*it)->m_type,
                              (*it)->m_name,
                              (*it)->m_client,
                              (*it)->m_port,
                              (*it)->m_duplex);
    }
}

// Create a local ALSA port for reference purposes
// and create a GUI Instrument for transmission upwards.
//
void
AlsaDriver::addInstrumentsForPort(Instrument::InstrumentType type,
                                  const std::string &name, 
                                  int client,
                                  int port,
                                  bool duplex)
{
    // only increment device number if we're on a new client
    //
    if (client != m_currentPair.first && m_currentPair.first != -1)
        m_deviceRunningId++;
 
    AlsaPort *alsaInstr;
    MappedInstrument *instr;
    std::string channelName;
    char number[100];


    if (type == Instrument::Midi)
    {

        // If we haven't added a metronome then add one to the first
        // instrument we add.   This is accomplished by adding a
        // MappedInstrument for Instrument #0
        //
        if(m_addedMetronome == false)
        {
            alsaInstr = new AlsaPort(0,
                                     0,
                                     name,
                                     client,
                                     port,
                                     duplex);  // a duplex port?

            m_alsaPorts.push_back(alsaInstr);

            for (int channel = 0; channel < 16; ++channel)
            {
                sprintf(number, " #%d", channel);
                channelName = "Metronome" + std::string(number);
                instr = new MappedInstrument(type,
                                             9, // always the drum channel
                                             channel,
                                             channelName,
                                             m_deviceRunningId);
                m_instruments.push_back(instr);
            }
            m_addedMetronome = true;
        }

        // Create AlsaPort with the start and end MappedInstrument
        // indexes.
        //
        alsaInstr = new AlsaPort(m_midiRunningId,
                                 m_midiRunningId + 15,
                                 name,
                                 client,
                                 port,
                                 duplex);  // a duplex port?

        m_alsaPorts.push_back(alsaInstr);

        for (int channel = 0; channel < 16; ++channel)
        {
            // Create MappedInstrument for export to GUI
            //
            sprintf(number, " #%d", channel + 1);
            channelName = name + std::string(number);

            if (channel == 9)
                channelName = name + std::string(" #10[D]");
            MappedInstrument *instr = new MappedInstrument(type,
                                                           channel,
                                                           m_midiRunningId++,
                                                           channelName,
                                                           m_deviceRunningId);
            m_instruments.push_back(instr);
        }

        MappedDevice *device =
                new MappedDevice(m_deviceRunningId,
                                 Rosegarden::Device::Midi,
                                 name,
                                 duplex);
        m_devices.push_back(device);
    }

    // Store these numbers for next time through
    m_currentPair.first = client;
    m_currentPair.second = port;
}
*/

// Set up queue, client and port

void Midi_Device_Alsa::Initialize()
{ 
    int result;

    // Create a non-blocking handle.
    // ("hw" will possibly give in to other handles in future?)
    
    if ( snd_seq_open( &handle, "hw", SND_SEQ_OPEN_DUPLEX, SND_SEQ_NONBLOCK) < 0)
     {
        fprintf( stderr, "Midi_Device_Alsa::Initialization: can't open sequencer:\n%s", snd_strerror(errno));
        return;
     }

    // Now we have a handle generate some possible destinations
    // through the Instrument metaphor
    
    //generateInstruments();


    // Create a queue
    
    //if((m_queue = snd_seq_alloc_named_queue(m_midiHandle,
    //                                            "Rosegarden queue")) < 0)
    //{
   //     std::cerr << "AlsaDriver::initialiseMidi - can't allocate queue"
    //              << std::endl;
    //    return;
    //}


    // Create a client
   
   snd_seq_set_client_name( handle, "KARAOKE");
   
   if(( client = snd_seq_client_id(handle)) < 0)
     {
        fprintf( stderr, "Midi_Device_Alsa::Initialization: can't create client\n");
        return;
     }
   
   // Create a port
   //
   port = snd_seq_create_simple_port(handle,
				     NULL,
				     SND_SEQ_PORT_CAP_WRITE |
				     SND_SEQ_PORT_CAP_SUBS_WRITE |
				     SND_SEQ_PORT_CAP_READ |
				     SND_SEQ_PORT_CAP_SUBS_READ,
				     SND_SEQ_PORT_TYPE_MIDI_GENERIC);
   if ( port < 0)
     {
        fprintf( stderr, "Midi_Device_Alsa::Initialization: can't create port\n");
        return;
     }
   
//   ClientPortPair inputDevice = getFirstDestination(true); // duplex = true
   
   std::cout << "    Record client set to (" << inputDevice.first
     << ", "
              << inputDevice.second
     << ")" << std::endl << std::endl;
   
   std::vector<AlsaPort*>::iterator it;
   
    // Connect to all available output client/ports
   //
   for (it = m_alsaPorts.begin(); it != m_alsaPorts.end(); it++)
     {
        if (snd_seq_connect_to(m_midiHandle,
                               m_port,
                               (*it)->m_client,
                               (*it)->m_port) < 0)
        {
            /*
            std::cerr << "AlsaDriver::initialiseMidi - "
                      << "can't subscribe output client/port ("
                      << (*it)->m_client << ", "
                      << (*it)->m_port << ")"
                      << std::endl;
                      */
        }
    }

    // Connect input port - enabling timestamping on the way through.
    // We have to fill out the subscription information as follows:
    
    snd_seq_addr_t sender, dest;
    snd_seq_port_subscribe_t *subs;
    snd_seq_port_subscribe_alloca(&subs);

    sender.client = inputDevice.first;
    sender.port = inputDevice.second;
    dest.client = m_client;
    dest.port = m_port;

    snd_seq_port_subscribe_set_sender(subs, &sender);
    snd_seq_port_subscribe_set_dest(subs, &dest);

    snd_seq_port_subscribe_set_queue(subs, m_queue);

    // enable time-stamp-update mode 
    
    snd_seq_port_subscribe_set_time_update(subs, 1);

    // set so we realtime timestamps
    
    snd_seq_port_subscribe_set_time_real(subs, 1);

    if (snd_seq_subscribe_port(m_midiHandle, subs) < 0)
    {
        std::cerr << "AlsaDriver::initialiseMidi - "
                  << "can't subscribe input client/port"
                  << std::endl;
        // Not the end of the world if this fails but we
        // have to flag it internally.
        
        m_midiInputPortConnected = false;
    }
    else
        m_midiInputPortConnected = true;

    // Erm?
    
    if (snd_seq_set_client_pool_output(m_midiHandle, 200) < 0 ||
        snd_seq_set_client_pool_input(m_midiHandle, 20) < 0 ||
        snd_seq_set_client_pool_output_room(m_midiHandle, 20) < 0)
    {
        std::cerr << "AlsaDriver::initialiseMidi - "
                  << "can't modify pool parameters"
                  << std::endl;
        return;
    }

    getSystemInfo();

    // Modify status with MIDI success
    
    m_driverStatus |= MIDI_OK;

    // Start the timer
    if ((result = snd_seq_start_queue(m_midiHandle, m_queue, NULL)) < 0)
    {
        std::cerr << "AlsaDriver::initialiseMidi - couldn't start queue - "
                  << snd_strerror(result)
                  << std::endl;
        exit(EXIT_FAILURE);
    }

    // process anything pending
    snd_seq_drain_output(m_midiHandle);

    std::cout << "AlsaDriver::initialiseMidi -  initialised MIDI subsystem"
              << std::endl;
}

/*
void AlsaDriver::initialisePlayback(const RealTime &position)
{
    std::cout << "AlsaDriver - initialisePlayback" << std::endl;
    m_alsaPlayStartTime = getAlsaTime();
    m_playStartPosition = position;
    m_startPlayback = true;
}


void AlsaDriver::stopPlayback()
{
    allNotesOff();
    m_playing = false;

    // send sounds-off to all client port pairs
    //
    std::vector<AlsaPort*>::iterator it;
    for (it = m_alsaPorts.begin(); it != m_alsaPorts.end(); ++it)
    {
        sendDeviceController(ClientPortPair((*it)->m_client,
                                            (*it)->m_port),
                             MIDI_CONTROLLER_SOUNDS_OFF,
                             0);
    }

    // Close any recording file
    if (m_recordStatus == RECORD_AUDIO && _recordFile)
    {
        _recordFile->close();

        // Create event to return to gui to say that we've completed
        // an audio file and we can generate a preview for it now.
        //
        try
        {
            MappedEvent *mE =
                new MappedEvent(_recordFile->getId(),
                                MappedEvent::AudioGeneratePreview,
                                0);

            // send completion event
            insertMappedEventForReturn(mE);
        }
        catch(...) {;}

        _recordFile = 0;
        m_recordStatus = ASYNCHRONOUS_AUDIO;

    }
        

    // Change recorded state if any set
    //
    if (m_recordStatus == RECORD_MIDI)
        m_recordStatus = ASYNCHRONOUS_MIDI;


    // Sometimes we don't "process" again before we actually
    // stop

}


void
AlsaDriver::resetPlayback(const RealTime &position, const RealTime &latency)
{
    // Reset note offs to correct positions
    //
    RealTime modifyNoteOff = m_playStartPosition - m_alsaPlayStartTime;

    // set new
    m_playStartPosition = position;
    m_alsaPlayStartTime = getAlsaTime() - latency;

    // add
    modifyNoteOff = modifyNoteOff - m_playStartPosition + m_alsaPlayStartTime;

    // modify the note offs that exist as they're relative to the
    // playStartPosition terms.
    //
    for (NoteOffQueue::iterator i = m_noteOffQueue.begin();
                                i != m_noteOffQueue.end(); ++i)
    {

        // if we're fast forwarding then we bring the note off closer
        if (modifyNoteOff <= RealTime(0, 0))
        {
            (*i)->setRealTime((*i)->getRealTime() + modifyNoteOff);
        }
        else // we're rewinding - kill the note immediately
        {
            (*i)->setRealTime(m_playStartPosition);
        }
    }
}


void
AlsaDriver::allNotesOff()
{
    snd_seq_event_t *event = new snd_seq_event_t();
    ClientPortPair outputDevice;
    RealTime offTime;

    // drop any pending notes
    snd_seq_drop_output_buffer(m_midiHandle);
    snd_seq_drop_output(m_midiHandle);

    // prepare the event
    snd_seq_ev_clear(event);
    snd_seq_ev_set_source(event, m_port);
    offTime = getAlsaTime();

    for (NoteOffQueue::iterator it = m_noteOffQueue.begin();
                                it != m_noteOffQueue.end(); ++it)
    {
        // Set destination according to instrument mapping to port
        //
        outputDevice = getPairForMappedInstrument((*it)->getInstrument());

        snd_seq_ev_set_dest(event,
                            outputDevice.first,
                            outputDevice.second);


        snd_seq_real_time_t alsaOffTime = { offTime.sec,
                                            offTime.usec * 1000 };

        snd_seq_ev_schedule_real(event, m_queue, 0, &alsaOffTime);
        snd_seq_ev_set_noteoff(event,
                               (*it)->getChannel(),
                               (*it)->getPitch(),
                               127);
        //snd_seq_event_output(m_midiHandle, event);
        snd_seq_event_output_direct(m_midiHandle, event);
        delete(*it);
    }
    
    m_noteOffQueue.erase(m_noteOffQueue.begin(), m_noteOffQueue.end());

    
    //std::cout << "AlsaDriver::allNotesOff - "
    //          << " queue size = " << m_noteOffQueue.size() << std::endl;
              

    // flush
    snd_seq_drain_output(m_midiHandle);
    delete event;
}

void
AlsaDriver::processNotesOff(const RealTime &time)
{
    static snd_seq_event_t event;

    ClientPortPair outputDevice;
    RealTime offTime;

    // prepare the event
    snd_seq_ev_clear(&event);
    snd_seq_ev_set_source(&event, m_port);

    NoteOffQueue::iterator it = m_noteOffQueue.begin();

    for (;it != m_noteOffQueue.end() && (*it)->getRealTime() <= time; ++it)
    {
        // Set destination according to instrument mapping to port
        //
        outputDevice = getPairForMappedInstrument((*it)->getInstrument());
        snd_seq_ev_set_dest(&event,
                            outputDevice.first,
                            outputDevice.second);

        offTime = (*it)->getRealTime();

        snd_seq_real_time_t alsaOffTime = { offTime.sec,
                                            offTime.usec * 1000 };

        snd_seq_ev_schedule_real(&event, m_queue, 0, &alsaOffTime);
        snd_seq_ev_set_noteoff(&event,
                               (*it)->getChannel(),
                               (*it)->getPitch(),
                               127);
        // send note off
        snd_seq_event_output(m_midiHandle, &event);
        delete(*it);
        m_noteOffQueue.erase(it);
    }

    // and flush them
    snd_seq_drain_output(m_midiHandle);

    
    //  std::cout << "AlsaDriver::processNotesOff - "
    //  << " queue size = " << m_noteOffQueue.size() << std::endl;
    
}

void
AlsaDriver::processAudioQueue(const RealTime &playLatency, bool now)
{
    std::vector<PlayableAudioFile*>::iterator it;
    RealTime currentTime = getSequencerTime() - playLatency;

    for (it = m_audioPlayQueue.begin(); it != m_audioPlayQueue.end(); ++it)
    {
        if ((currentTime >= (*it)->getStartTime() || now) &&
            (*it)->getStatus() == PlayableAudioFile::IDLE)
        {
            (*it)->setStatus(PlayableAudioFile::PLAYING);
        }

        if (currentTime >= (*it)->getEndTime() &&
            (*it)->getStatus() == PlayableAudioFile::PLAYING)
        {
             (*it)->setStatus(PlayableAudioFile::DEFUNCT);

             // Simple event to inform that AudioFileId has
             // now stopped playing.
             //
             try
             {
                 MappedEvent *mE =
                     new MappedEvent((*it)->getAudioFile()->getId(),
                                     MappedEvent::AudioStopped,
                                     0);

                 // send completion event
                 insertMappedEventForReturn(mE);
             }
             catch(...) {;}
        }
    }

}

// Get the queue time and convert it to RealTime for the gui
// to use.
//
RealTime
AlsaDriver::getSequencerTime()
{
    if(m_playing)
       return getAlsaTime() + m_playStartPosition - m_alsaPlayStartTime;

    return RealTime(0, 0);
}

// Gets the time of the ALSA queue
//
RealTime
AlsaDriver::getAlsaTime()
{
    RealTime sequencerTime(0, 0);

    snd_seq_queue_status_t *status;
    snd_seq_queue_status_malloc(&status);

    if (snd_seq_get_queue_status(m_midiHandle, m_queue, status) < 0)
    {
        std::cerr << "AlsaDriver::getSequencerTime - can't get queue status"
                  << std::endl;
        return sequencerTime;
    }

    sequencerTime.sec = snd_seq_queue_status_get_real_time(status)->tv_sec;

    double microSeconds = snd_seq_queue_status_get_real_time(status)->tv_nsec
                          /1000.0;

    sequencerTime.usec = (int)microSeconds;

    snd_seq_queue_status_free(status);

    return sequencerTime;
}


// Get all pending input events and turn them into a MappedComposition.
//
//
/*MappedComposition*
AlsaDriver::getMappedComposition(const RealTime &playLatency)
{
    // If we're already inserted some audio VU meter events then
    // don't clear them from the composition.
    //
    if (m_audioMeterSent == false)
        m_recordComposition.clear();


    if (m_recordStatus != RECORD_MIDI &&
        m_recordStatus != RECORD_AUDIO &&
        m_recordStatus != ASYNCHRONOUS_MIDI &&
        m_recordStatus != ASYNCHRONOUS_AUDIO)
           return &m_recordComposition;

    // If the input port hasn't connected we shouldn't poll it
    //
    if(m_midiInputPortConnected == false)
        return &m_recordComposition;

    Rosegarden::RealTime eventTime(0, 0);

    snd_seq_event_t *event;

    while(snd_seq_event_input(m_midiHandle, &event) > 0)
    {
        unsigned int channel = (unsigned int)event->data.note.channel;
        unsigned int chanNoteKey = ( channel << 8 ) +
                                   (unsigned int) event->data.note.note;

        eventTime.sec = event->time.time.tv_sec;
        eventTime.usec = event->time.time.tv_nsec / 1000;
        eventTime = eventTime - m_alsaRecordStartTime + m_playStartPosition
                              - playLatency;

        switch(event->type)
        {

            case SND_SEQ_EVENT_NOTE:
            case SND_SEQ_EVENT_NOTEON:
                if (event->data.note.velocity > 0)
                {
                    m_noteOnMap[chanNoteKey] = new MappedEvent();
                    m_noteOnMap[chanNoteKey]->setPitch(event->data.note.note);
                    m_noteOnMap[chanNoteKey]->
                        setVelocity(event->data.note.velocity);
                    m_noteOnMap[chanNoteKey]->setEventTime(eventTime);

                    // Negative duration - we need to hear the NOTE ON
                    // so we must insert it now with a negative duration
                    // and pick and mix against the following NOTE OFF
                    // when we create the recorded segment.
                    //
                    m_noteOnMap[chanNoteKey]->setDuration(RealTime(-1, 0));

                    // Create a copy of this when we insert the NOTE ON -
                    // keeping a copy alive on the m_noteOnMap.
                    //
                    // We shake out the two NOTE Ons after we've recorded
                    // them.
                    //
                    m_recordComposition.insert(
                            new MappedEvent(m_noteOnMap[chanNoteKey]));

                    break;
                }

            case SND_SEQ_EVENT_NOTEOFF:
                if (m_noteOnMap[chanNoteKey] != 0)
                {
                    // Set duration correctly on the NOTE OFF
                    //
                    RealTime duration = eventTime -
                             m_noteOnMap[chanNoteKey]->getEventTime();

                    assert(duration >= RealTime(0, 0));

                    // Velocity 0 - NOTE OFF.  Set duration correctly
                    // for recovery later.
                    //
                    m_noteOnMap[chanNoteKey]->setVelocity(0);
                    m_noteOnMap[chanNoteKey]->setDuration(duration);

                    
                    //std::cout << "Inserted NOTE at "
                    //          << m_noteOnMap[chanNoteKey]->getEventTime()
                    //          << " with duration "
                    //          << m_noteOnMap[chanNoteKey]->getDuration()
                    //          << std::endl;
                    

                    // force shut off of note
                    m_recordComposition.insert(m_noteOnMap[chanNoteKey]);

                    // reset the reference
                    //
                    m_noteOnMap[chanNoteKey] = 0;

                }
                break;

            case SND_SEQ_EVENT_KEYPRESS:
                {
                    MappedEvent *mE = new MappedEvent();
                    mE->setType(MappedEvent::MidiKeyPressure);
                    mE->setEventTime(eventTime);
                    mE->setData1(event->data.control.value >> 7);
                    mE->setData2(event->data.control.value & 0x7f);
                    m_recordComposition.insert(mE);
                }
                break;

            case SND_SEQ_EVENT_CONTROLLER:
                {
                    MappedEvent *mE = new MappedEvent();
                    mE->setType(MappedEvent::MidiController);
                    mE->setEventTime(eventTime);
                    mE->setData1(event->data.control.param);
                    mE->setData2(event->data.control.value);
                    m_recordComposition.insert(mE);
                }
                break;

            case SND_SEQ_EVENT_PGMCHANGE:
                {
                    MappedEvent *mE = new MappedEvent();
                    mE->setType(MappedEvent::MidiProgramChange);
                    mE->setEventTime(eventTime);
                    mE->setData1(event->data.control.value);
                    m_recordComposition.insert(mE);

                }
                break;

            case SND_SEQ_EVENT_PITCHBEND:
                {
                    MappedEvent *mE = new MappedEvent();
                    mE->setType(MappedEvent::MidiPitchBend);
                    mE->setEventTime(eventTime);
                    mE->setData1(event->data.control.value >> 7);
                    mE->setData2(event->data.control.value & 0x7f);
                    m_recordComposition.insert(mE);
                }
                break;

            case SND_SEQ_EVENT_CHANPRESS:
                {
                    MappedEvent *mE = new MappedEvent();
                    mE->setType(MappedEvent::MidiChannelPressure);
                    mE->setEventTime(eventTime);
                    mE->setData1(event->data.control.value >> 7);
                    mE->setData2(event->data.control.value & 0x7f);
                    m_recordComposition.insert(mE);
                }
               break;

            case SND_SEQ_EVENT_SYSEX:
               {
                   // Bundle up the data into a block on the MappedEvent
                   //
                   std::string data;
                   char *ptr = (char*)(event->data.ext.ptr);
                   for (unsigned int i = 0; i < event->data.ext.len; ++i)
                       data += *(ptr++);

                   MappedEvent *mE = new MappedEvent();
                   mE->setType(MappedEvent::MidiSystemExclusive);
                   mE->setDataBlock(data);
                   m_recordComposition.insert(mE);

               }
               break;


            case SND_SEQ_EVENT_SENSING:
               // MIDI device is still there
               break;

            default:
               std::cerr << "AlsaDriver::getMappedComposition - "
                         << "got unrecognised MIDI event type = "
                         << int(event->type) << std::endl;
               break;


        }

        snd_seq_free_event(event);
    }

    // reset this whatever
    //
    m_audioMeterSent = false;

    return &m_recordComposition;
}
  

*/		      
     
   
   
/////////////////////////////// JCMODIF /////////////////////////////
   /*
void
AlsaDriver::processMidiOut(const MappedComposition &mC,
                           const RealTime &playLatency,
                           bool now)
{
    Rosegarden::RealTime midiRelativeTime;
    Rosegarden::RealTime midiRelativeStopTime;
    Rosegarden::MappedInstrument *instrument;
    ClientPortPair outputDevice;
    MidiByte channel;
    snd_seq_event_t *event = new snd_seq_event_t();

    // These won't change in this slice
    //
    snd_seq_ev_clear(event);
    snd_seq_ev_set_source(event, m_port);

    for (MappedComposition::iterator i = mC.begin(); i != mC.end(); ++i)
    {
        if ((*i)->getType() == MappedEvent::Audio)
            continue;


        midiRelativeTime = (*i)->getEventTime() - m_playStartPosition +
                           playLatency + m_alsaPlayStartTime;

        // Second and nanoseconds for ALSA
        //
        snd_seq_real_time_t time = { midiRelativeTime.sec,
                                     midiRelativeTime.usec * 1000 };

        // millisecond note duration
        //
        unsigned int eventDuration = (*i)->getDuration().sec * 1000
             + (*i)->getDuration().usec / 1000;

        // Set destination according to Instrument mapping
        //
        outputDevice = getPairForMappedInstrument((*i)->getInstrument());

        snd_seq_ev_set_dest(event,
                            outputDevice.first,
                            outputDevice.second);

        
       // cout << "INSTRUMENT = " << (*i)->getInstrument() << endl;
//
  //      cout << "TIME = " << time.tv_sec << " : " << time.tv_nsec * 1000
    //          << endl;
//
//
  //      std::cout << "EVENT to " << (int)event->dest.client
    //              << " : " 
      //            << (int)event->dest.port << endl;
        

        snd_seq_ev_schedule_real(event, m_queue, 0, &time);
        instrument = getMappedInstrument((*i)->getInstrument());

        // set the stop time for Note Off
        //
        midiRelativeStopTime = midiRelativeTime + (*i)->getDuration();
 
        if (instrument != 0)
            channel = instrument->getChannel();
        else
        {
            std::cerr << "processMidiOut() - couldn't get Instrument for Event"
                      << std::endl;
            channel = 0;
        }

        switch((*i)->getType())
        {
            case MappedEvent::MidiNoteOneShot:
                {
                    // Just an arbitrary duration for one-shot Notes
                    // for the moment until we work out the timing
                    // conversion.
                    //
                    int duration = 100;
                    snd_seq_ev_set_note(event,
                                        channel,
                                        (*i)->getPitch(),
                                        (*i)->getVelocity(),
                                        duration);
                }
                break;

            case MappedEvent::MidiNote:
                // If we've got an "infinite" note then just noteon -
                // else send the duration.
                //
                if ((*i)->getDuration() == Rosegarden::RealTime(-1, 0))
                {
                    snd_seq_ev_set_noteon(event,
                                          channel,
                                          (*i)->getPitch(),
                                          (*i)->getVelocity());
                }
                else
                {
                    if ((*i)->getVelocity() == 0)
                    {
                        snd_seq_ev_set_noteoff(event,
                                               channel,
                                               (*i)->getPitch(),
                                               (*i)->getVelocity());
                    }
                    else
                    {
                        snd_seq_ev_set_note(event,
                                            channel,
                                            (*i)->getPitch(),
                                            (*i)->getVelocity(),
                                            eventDuration);
                    }
                }
                break;

            case MappedEvent::MidiProgramChange:
                snd_seq_ev_set_pgmchange(event,
                                         channel,
                                         (*i)->getData1());
                break;

            case MappedEvent::MidiKeyPressure:
                snd_seq_ev_set_keypress(event,
                                        channel,
                                        (*i)->getData1(),
                                        (*i)->getData2());
                break;

            case MappedEvent::MidiChannelPressure:
                snd_seq_ev_set_chanpress(event,
                                         channel,
                                         (*i)->getData1());
                break;

            case MappedEvent::MidiPitchBend:
                {
                    int value = ((*i)->getData1() << 7) |
                                ((*i)->getData2() & 0x7F);

                    // keep within -8192 to +8192
                    //
                    if (value & 0x4000)
                        value -= 0x8000;

                    snd_seq_ev_set_pitchbend(event,
                                             channel,
                                             value);
                }
                break;

            case MappedEvent::MidiSystemExclusive:
                {
                    // pack data between start and end blocks
                    //

                    char out[2];
                    sprintf(out, "%c", MIDI_SYSTEM_EXCLUSIVE);
                    std::string data = out;

                    data += (*i)->getDataBlock();

                    sprintf(out, "%c", MIDI_END_OF_EXCLUSIVE);
                    data += out;

                    snd_seq_ev_set_sysex(event,
                                         data.length(),
                                         (char*)(data.c_str()));
                }
                break;

            case MappedEvent::MidiController:
                snd_seq_ev_set_controller(event,
                                          channel,
                                          (*i)->getData1(),
                                          (*i)->getData2());
                break;

            case MappedEvent::Audio:
            case MappedEvent::AudioCancel:
            case MappedEvent::AudioLevel:
            case MappedEvent::AudioStopped:
                break;

            default:
                std::cout << "AlsaDriver::processMidiOut - "
                          << "unrecognised event type"
                          << std::endl;
                delete event;
                snd_seq_drain_output(m_midiHandle);
                return;
        }

        if (now || m_playing == false)
        {
            RealTime nowTime = getAlsaTime();
            snd_seq_real_time_t outTime = { nowTime.sec,
                                         nowTime.usec * 1000 };
            snd_seq_ev_schedule_real(event, m_queue, 0, &outTime);
            snd_seq_event_output_direct(m_midiHandle, event);
        }
        else
            snd_seq_event_output(m_midiHandle, event);

        // Add note to note off stack
        //
        if ((*i)->getType() == MappedEvent::MidiNote)
        {
            bool extended = false;
            NoteOffQueue::iterator it;

            for (it = m_noteOffQueue.begin(); it != m_noteOffQueue.end(); ++it)
            {
                if ((*it)->getPitch() == (*i)->getPitch() &&
                    (*it)->getChannel() == channel &&
                    (*it)->getInstrument() == (*i)->getInstrument())
                {
                    (*it)->setRealTime(midiRelativeStopTime);
                    extended = true;
                }
            }

            if (!extended)
            {
                NoteOffEvent *noteOffEvent =
                    new NoteOffEvent(midiRelativeStopTime, // already calculated
                                     (*i)->getPitch(),
                                     channel,
                                     (*i)->getInstrument());
                m_noteOffQueue.insert(noteOffEvent);
            }
        }
    }

    snd_seq_drain_output(m_midiHandle);

    delete event;
}


// This is almost identical to the aRts driver version at
// the moment.  Can't see it ever having to change that
// much really.
//
void
AlsaDriver::processEventsOut(const MappedComposition &mC,
                             const Rosegarden::RealTime &playLatency,
                             bool now)
{
    if (m_startPlayback)
    {
        m_alsaPlayStartTime = getAlsaTime();
        m_startPlayback= false;
        m_playing = true;
    }

    AudioFile *audioFile = 0;

    // insert audio events if we find them
    for (MappedComposition::iterator i = mC.begin(); i != mC.end(); ++i)
    {
        // Play an audio file
        //
        if ((*i)->getType() == MappedEvent::Audio)
        {
            // Check for existence of file - if the sequencer has died
            // and been restarted then we're not always loaded up with
            // the audio file references we should have.  In the future
            // we could make this just get the gui to reload our files
            // when (or before) this fails.
            //
            audioFile = getAudioFile((*i)->getAudioID());

            if (audioFile)
            { 
                Rosegarden::RealTime adjustedEventTime =
                    (*i)->getEventTime();

                
                  //  adjustedEventTime == RealTime(0, 0) &&
                  //  getSequencerTime() > RealTime(1, 0))
                    

                // If we're playing, the event time is two minutes or
                // more in the past we've sent an async audio event -
                // if we're playing we have to reset this time to
                // some point in our playing future - otherwise we
                // just reset to zero.
                //
                if (adjustedEventTime <= RealTime(-120, 0))
                {
                    if (m_playing)
                        adjustedEventTime = getAlsaTime() + RealTime(0, 500000);
                    else
                        adjustedEventTime = RealTime(0, 0);
                }


                PlayableAudioFile *audioFile =
                    new PlayableAudioFile((*i)->getInstrument(),
                                          getAudioFile((*i)->getAudioID()),
                                          adjustedEventTime - playLatency,
                                          (*i)->getAudioStartMarker(),
                                          (*i)->getDuration());

                // If the start index is non-zero then we must scan to
                // the required starting position in the sample
                if (audioFile->getStartIndex() != RealTime(0, 0))
                {
                    // If there's a problem with the scan (most likely
                    // we've moved beyond the end of the sample) then
                    // cast it away.
                    //
                    if(audioFile->scanTo(audioFile->getStartIndex()) == false)
                    {
                        std::cerr << "AlsaDriver::processEventsOut - "
                                  << "skipping audio file" << std::endl;
                        delete audioFile;
                        continue;
                    }
                }

                queueAudio(audioFile);
            }
            else
            {
                std::cerr << "AlsaDriver::processEventsOut - "
                          << "can't find audio file reference" 
                          << std::endl;

                std::cerr << "AlsaDriver::processEventsOut - "
                          << "try reloading the current Rosegarden file"
                          << std::endl;
            }
        }

        // Cancel a playing audio file
        //
        if ((*i)->getType() == MappedEvent::AudioCancel)
        {
            cancelAudioFile(Rosegarden::InstrumentId((*i)->getInstrument()),
                             Rosegarden::AudioFileId((*i)->getData1()));
        }
    }

    // Process Midi and Audio
    //
    processMidiOut(mC, playLatency, now);
    processAudioQueue(playLatency, now);

}


void
AlsaDriver::record(const RecordStatus& recordStatus)
{
    if (recordStatus == RECORD_MIDI)
    {
        // start recording
        m_recordStatus = RECORD_MIDI;
        m_alsaRecordStartTime = getAlsaTime();
    }
    else if (recordStatus == RECORD_AUDIO)
    {
	std::cerr << "AlsaDriver::record - can't record audio without JACK"
		  << std::endl;

    }
    else
    if (recordStatus == ASYNCHRONOUS_MIDI)
    {
        m_recordStatus = ASYNCHRONOUS_MIDI;
    }
    else if (recordStatus == ASYNCHRONOUS_AUDIO)
    {
        m_recordStatus = ASYNCHRONOUS_AUDIO;
    }
    else
    {
        std::cerr << "ArtsDriver::record - unsupported recording mode"
                  << std::endl;
    }
}

ClientPortPair
AlsaDriver::getFirstDestination(bool duplex)
{
    ClientPortPair destPair(-1, -1);
    std::vector<AlsaPort*>::iterator it;

    for (it = m_alsaPorts.begin(); it != m_alsaPorts.end(); ++it)
    {
        destPair.first = (*it)->m_client;
        destPair.second = (*it)->m_port;

        // If duplex port is required then choose first one
        //
        if (duplex)
        {
            if ((*it)->m_duplex == true)
                return destPair;
        }
        else
        {
            // If duplex port isn't required then choose first
            // specifically non-duplex port (should be a synth)
            //
            if ((*it)->m_duplex == false)
                return destPair;
        }
    }

    return destPair;
}


// Sort through the ALSA client/port pairs for the range that
// matches the one we're querying.  If none matches then send
// back -1 for each.
//
ClientPortPair
AlsaDriver::getPairForMappedInstrument(InstrumentId id)
{
    ClientPortPair matchPair(-1, -1);

    std::vector<AlsaPort*>::iterator it;

    for (it = m_alsaPorts.begin(); it != m_alsaPorts.end(); ++it)
    {
        if (id >= (*it)->m_startId && id <= (*it)->m_endId)
        {
            matchPair.first = (*it)->m_client;
            matchPair.second = (*it)->m_port;
            return matchPair;
        }
    }

    return matchPair;
}

// Send a direct controller to the specified port/client
//
void
AlsaDriver::sendDeviceController(const ClientPortPair &device,
                                 MidiByte controller,
                                 MidiByte value)
{
    snd_seq_event_t *event = new snd_seq_event_t();


    // These won't change in this slice
    //
    snd_seq_ev_clear(event);
    snd_seq_ev_set_source(event, m_port);

    snd_seq_ev_set_dest(event,
                        device.first,
                        device.second);

    for (int i = 0; i < 16; i++)
    {
        snd_seq_ev_set_controller(event,
                                  i,
                                  controller,
                                  value);
        snd_seq_event_output_direct(m_midiHandle, event);
    }

    snd_seq_drain_output(m_midiHandle);

    delete event;
}

// We only process note offs in this section
//
void
AlsaDriver::processPending(const RealTime &playLatency)
{
    if (!m_playing)
        processAudioQueue(playLatency, true);
    else
        processNotesOff(getAlsaTime() + playLatency);
}

void
AlsaDriver::insertMappedEventForReturn(MappedEvent *mE)
{
    // If we haven't inserted a MappedEvent yet this update period
    // then clear down the composition and flag
    //
    if (m_audioMeterSent == false)
    {
        m_recordComposition.clear();
        m_audioMeterSent = true;
    }

    // Insert the event ready for return at the next opportunity
    //
    m_recordComposition.insert(mE);
}

void
AlsaDriver::setPluginInstance(InstrumentId id,
                              unsigned long pluginId,
                              int position)
{

}


void
AlsaDriver::removePluginInstance(InstrumentId id, int position)
{
}

void
AlsaDriver::setPluginInstancePortValue(InstrumentId id,
                                       int position,
                                       unsigned long portNumber,
                                       float value)
{
} 



// Return the sample rate of the JACK driver if we have one installed
//
unsigned int
AlsaDriver::getSampleRate() const
{
   return 0;
}


// At some point make this check for just different numbers of clients
//
bool
AlsaDriver::checkForNewClients()
{
    snd_seq_client_info_t *cinfo;
    snd_seq_port_info_t *pinfo;
    int  client;
    unsigned int cap;
    unsigned int currentClientCount = 0,
                 oldClientCount = 0;

    snd_seq_client_info_alloca(&cinfo);
    snd_seq_client_info_set_client(cinfo, -1);

    std::vector<MappedDevice*>::iterator it = m_devices.begin();
    for (; it != m_devices.end(); ++it)
    {
        if ((*it)->getType() == Rosegarden::Device::Midi)
            oldClientCount++;
    }

    // Count current clients
    //
    while (snd_seq_query_next_client(m_midiHandle, cinfo) >= 0)
    {
        client = snd_seq_client_info_get_client(cinfo);
        snd_seq_port_info_alloca(&pinfo);
        snd_seq_port_info_set_client(pinfo, client);
        snd_seq_port_info_set_port(pinfo, -1);

        // ignore ourselves
        if (m_client == client) continue;

        while (snd_seq_query_next_port(m_midiHandle, pinfo) >= 0)
        {
            cap = (SND_SEQ_PORT_CAP_SUBS_WRITE|SND_SEQ_PORT_CAP_WRITE);

            if ((snd_seq_port_info_get_capability(pinfo) & cap) == cap)
                currentClientCount++;
        }
    }

    // Have we got a different number of clients than we had before?
    //
    if (oldClientCount != currentClientCount)
    {
        generateInstruments();

        MappedEvent *mE =
            new MappedEvent(0, MappedEvent::SystemUpdateInstruments,
                            0, 0);
        // send completion event
        insertMappedEventForReturn(mE);
        return true;
    }

    return false;
}

void
AlsaDriver::setPluginInstanceBypass(InstrumentId id,
                                    int position,
                                    bool value)
{
    std::cout << "AlsaDriver::setPluginInstanceBypass - "
              << value << std::endl;

}

*/
