#include "gl_particles.hpp"
#include <stdio.h>
#include <stdlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <SDL.h>
#include <SDL_image.h>

#define MAX_PARTICLES         100
#define DEFAULT_SPEED        1.0f   // gl units per seconds
#define DEFAULT_DISPERSION   1.0f
#define DEFAULT_GRAVITY_X    0.0f
#define DEFAULT_GRAVITY_Y   -1.5f;//-0.8f
#define DEFAULT_GRAVITY_Z    0.0f
#define SPEED_FACTOR        10.0f

#ifndef min
# define min(x,y) ( (x)<=(y)?(x):(y))
#endif

extern GLuint VSDL_LoadTexture(char *filename);
 
struct Particle
{
   bool  active;
   float life;

   float fade;
   
   float r;      /* Red Value       */
   float g;      /* Green Value     */
   float b;      /* Blue Value      */
   
   float x;      /* X Position      */
   float y;      /* Y Position      */
   float z;      /* Z Position      */
   
   float xi;     /* X Direction     */
   float yi;     /* Y Direction     */
   float zi;     /* Z Direction     */
};


Gl_Particles::Gl_Particles( int nb_particlesn, char *texture)
{
   if ( nb_particles < 1) 
     nb_particles = 1;
   if ( nb_particles > MAX_PARTICLES)
     nb_particles = MAX_PARTICLES;
   
   this->nb_particles = nb_particles;
   particles = new Particle[nb_particles];
   speed = DEFAULT_SPEED;
   dispersion = DEFAULT_DISPERSION;
   start_x = 0.;
   start_y = 0.;
   start_z = 0.;
   start_speed_x=0.;
   start_speed_y=-1.;
   start_speed_z=0.;   
   gravity_x = 0.;
   gravity_y= -0.8;
   gravity_z = 0.;
   
   starting_position.x = 0.0f;
   starting_position.y = 0.0f;
   starting_position.z = 0.0f;
   
   starting_position.xi = 0.0f;
   starting_position.yi = 0.0f;
   starting_position.zi = 0.0f;
   
   starting_position.time_left = 0.0f;

   this->texture = VSDL_LoadTexture( "particle.bmp");
   
   Reset();
}

Gl_Particles::~Gl_Particles()
{
   glDeleteTextures( 1, &texture);
   delete particles;
}

void Gl_Particles::Reset()
{
   for( unsigned int p=0;p<nb_particles;p++)
     {
	LaunchParticle( p);
     }
}

void Gl_Particles::Draw()
{
   // Render each of the particles

   glMatrixMode( GL_MODELVIEW);
   glDisable( GL_LIGHTING);
   glShadeModel( GL_SMOOTH );
   glDisable( GL_DEPTH_TEST );
   glEnable( GL_BLEND );
   glBlendFunc( GL_SRC_ALPHA, GL_ONE );

   //fprintf( stderr, "is texture: %d (%d)\n", glIsTexture(this->texture),this->texture);
   glEnable( GL_TEXTURE_2D);
   glBindTexture( GL_TEXTURE_2D, this->texture);

   glBegin( GL_QUADS);
   glTexCoord2f( 0.0f,0.0f);
   glVertex3f( 0,0,0);
  
   glTexCoord2f( 0.0f,1.0f);
   glVertex3f( 0,20,0);
 
   glTexCoord2f( 1.0f,1.0f);
   glVertex3f( 20,20,0);
   
   glTexCoord2f( 1.0f,0.0f);
   glVertex3f( 20,0,0);
     glEnd();
   
   for (unsigned int p=0;p<nb_particles;p++)
     {
	if ( particles[p].active )
	  { 
	     float x = particles[p].x;
	     float y = particles[p].y;
	     float z = particles[p].z;
	     
	     // Draw The Particle Using Our RGB Values,
	     // Fade The Particle Based On It's Life
	     
	     glColor4f( particles[p].r,
			particles[p].g,
			particles[p].b,
			particles[p].life );
	     
	     // Draw a quad with triangle strip
	     glBegin( GL_TRIANGLE_STRIP);
	     // top right
	     glTexCoord2d( 1, 1 );
	     glVertex3f( x + 2., (y + 2.), z );
	     // top left
	     glTexCoord2d( 0, 1 );
	     glVertex3f( x - 2., (y + 2.), z );
	     // bottom right
	     glTexCoord2d( 1, 0 );
	     glVertex3f( x + 2., (y - 2.), z );
	     // bottom left
	     glTexCoord2d( 0, 0 );
	     glVertex3f( x - 2., (y - 2.), z );
	     glEnd( );
	  }
     }
   
   if ( glGetError() != 0 )
     fprintf( stderr, "GL_error!\n"); 
}

void Gl_Particles::Update( unsigned int dt)
{
   float dt_sec = (float) dt / 1000.0f;
   
   // update spreeding position 
   float t = min( dt_sec, starting_position.time_left);
   starting_position.time_left -= t;
   starting_position.x += starting_position.xi * t;
   starting_position.y += starting_position.yi * t;
   starting_position.z += starting_position.zi * t;
   
   for (unsigned int p=0;p<nb_particles;p++)
     {
	if ( particles[p].active )
	  {
	     // Update position (along speed)
	     particles[p].x += particles[p].xi * SPEED_FACTOR * dt_sec;
	     particles[p].y += particles[p].yi * SPEED_FACTOR * dt_sec;
	     particles[p].z += particles[p].zi * SPEED_FACTOR * dt_sec;
	     
	     // Update inerty (along gravity)
	     particles[p].xi += gravity_x * SPEED_FACTOR * dt_sec;
	     particles[p].yi += gravity_y * SPEED_FACTOR * dt_sec;
	     particles[p].zi += gravity_z * SPEED_FACTOR * dt_sec;
	     
	     // update life
	     particles[p].life -= particles[p].fade * dt_sec ;
	     
	     // If the particle dies, revive it
	     if ( particles[p].life < 0.0f )
	       LaunchParticle( p);
	  }
     }
}

void Gl_Particles::MoveTo( float x, float y, float z, float dt_sec)
{
   starting_position.xi = ( x - starting_position.x) / dt_sec;
   starting_position.yi = ( y - starting_position.y) / dt_sec;
   starting_position.zi = ( z - starting_position.z) / dt_sec;
   
   starting_position.time_left = dt_sec;
}


void Gl_Particles::LaunchParticle( int no_particle)
{  
   unsigned int p = no_particle;
   
   particles[p].active = true;
   particles[p].life = 0.1f + ((float)(rand()%900))/1000.0f ; // between 0.1 second and 2 seconds 
   // 1/fade = the number of updates before particle death
   // maxi 200 updates / mini 10 updates
   particles[p].fade = 1.0f;// + (float)(rand()%100)/10000.0f;
   particles[p].r = 0.9+(float)(rand()%100)/1000.0f;
   particles[p].g = 0.7+(float)(rand()%300)/1000.0f;
   particles[p].b = 0.7+(float)(rand()%300)/1000.0f;
   particles[p].x = starting_position.x;
   particles[p].y = starting_position.y;
   particles[p].z = starting_position.z;
   particles[p].xi = start_speed_x+(float)(rand()%4000)/1000.0f*dispersion;
   particles[p].yi = start_speed_y+(float)(rand()%4000)/1000.0f*dispersion;
   particles[p].zi = start_speed_z+(float)(rand()%4000)/1000.0f*dispersion;	  
}

GLuint VSDL_LoadTexture(char *filename)
{
   SDL_Surface *image; 
   GLuint texture_object[1];
   
   if ( ( image = SDL_LoadBMP( filename ) ) )
     {
		
	glGenTextures( 1, texture_object);
	
	glBindTexture( GL_TEXTURE_2D, texture_object[0] );
	
	glTexImage2D( GL_TEXTURE_2D, 0, 3, image->w,
		      image->h, 0, GL_BGR,
		      GL_UNSIGNED_BYTE, image->pixels );

	if ( glGetError() != 0 )
	  fprintf( stderr, "GL_error!\n"); 
	
	glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
 
	if ( glGetError() != 0 )
	  fprintf( stderr, "GL_error!\n"); 
	
	SDL_FreeSurface( image);
	return texture_object[0];
     }
   else
     {
	fprintf( stderr, "load_gl_textrue: can't read file %s\n", filename);
	return 0;
     }
}
