#include <stdlib.h>
#include <stdio.h>
#include <SDL.h>
#include <GL/gl.h>
#include "display_manager.hpp"
#include "display.hpp"

#define STATE_UNINITIALIZED 0
#define STATE_RUNNING       1
#define STATE_EXITING       2

DisplayManager::DisplayManager()
{
   current_display = NULL;
   thread = NULL;
   mutex = NULL;
   state = STATE_UNINITIALIZED;
}

DisplayManager::~DisplayManager()
{
   Terminate();
}

void DisplayManager::Initialize( char *path_to_display_plugins)
{

   state = STATE_RUNNING;
   
   mutex = SDL_CreateMutex();
   
   thread = SDL_CreateThread( LoopDisplay, this);
   
   fprintf( stdout, "DisplayManager::Initialize: Sucessfull\n");
}

void DisplayManager::Terminate()
{
   // 1) Stop thread
   state = STATE_EXITING;
   if ( thread )
     SDL_WaitThread( thread, NULL);
   if ( mutex)
     SDL_DestroyMutex( mutex);
   
   // 2) Unload current plugin
   if ( current_display)
     UnLoadDisplay( current_display);
   
   // 3) Close display window
   SDL_Quit();
}

void DisplayManager::LoadDisplayPlugin( char *plugin_name)
{  
   if ( current_display != NULL )
     UnLoadDisplay( current_display);
   
   current_display = LoadDisplay( plugin_name);

   if ( current_display != NULL)
     current_display->Start();
 
   if ( current_display)
     fprintf( stderr, "DisplayManager::LoadDisplayPlugin: plugin %s succesfully loaded.\n", plugin_name);
   else
     fprintf( stderr, "DisplayManager::LoadDisplayPlugin: error when loading plugin %s.\n", plugin_name);
}

void DisplayManager::Update( unsigned int current_tick)
{
   SDL_mutexP( mutex);

   if ( current_display != NULL)
     current_display->Update( current_tick);

   SDL_mutexV( mutex);
}
   
void DisplayManager::SetLyric( Midi_Lyric* lyric)
{
   SDL_mutexP( mutex);
   
   if ( current_display != NULL)
     current_display->SetLyric( lyric);   
   
   SDL_mutexV( mutex);
}

void DisplayManager::Loop()
{
   SDL_Surface *screen;
   int videoFlags = SDL_OPENGL;//|SDL_FULLSCREEN ;
   int width = 640;
   int height = 480;
   char *title = "KARAOKE_DISPLAY";
  
   // 1) Init OpenGL

   if ( SDL_Init(SDL_INIT_VIDEO) < 0 ) 
     {
	fprintf( stderr, "DisplayManager::Initialize: can't init video!\n");
	return;
     }
   
   SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
   
   if ( ( screen = SDL_SetVideoMode( width, height, 0, videoFlags)) == NULL ) 
     {
	fprintf( stderr, "DisplayManager::Initialize: can't set %dx%dxanybpp video mode: %s\n",
		 width, height, SDL_GetError());
	return;
     }
   
   SDL_WM_SetCaption( title, title) ;
   
   glViewport( 0, 0, screen->w, screen->h);
   
   glMatrixMode( GL_PROJECTION);
   glLoadIdentity();
   //gluPerspective( 45., 640./480., 0.1, 200);
     
   glMatrixMode( GL_MODELVIEW);
   glLoadIdentity();

   
   
   
   ////////////////
   
   Uint32 last_tick, current_tick, nb_ticks;
   
   current_tick = SDL_GetTicks();
   
   while( state == STATE_RUNNING)
     {	
	// Take mutex to avoid sequencer change lyrics while Refresh use them
	SDL_mutexP( mutex);
	
	//fprintf( stderr, "plugin display\n");

	last_tick = current_tick;
	current_tick = SDL_GetTicks();
	nb_ticks = current_tick-last_tick;
	//if ( nb_ticks < 1 ) nb_ticks = 1;
	
	if ( current_display != NULL)
	  current_display->Refresh( nb_ticks);
	
	//fprintf( stderr, "plugin displayed\n");
	
	SDL_mutexV( mutex);
	
	SDL_GL_SwapBuffers();
	
	//fprintf( stderr, "buffers swapped\n");
	
	//fprintf( stdout, "fps=%d\n", WaitFrame(100)); // 100 fps
        WaitFrame(100);
     }
}

unsigned int DisplayManager::WaitFrame( unsigned int fps) {
   static Uint32 next_tick = 0 ;
   Uint32 this_tick ;
   int realfps ;
   
   this_tick = SDL_GetTicks() ;
   if ( this_tick <= next_tick ) {
      SDL_Delay( next_tick-this_tick) ;
      realfps = fps ;
   }
   else 
     realfps = 1000/(1000/fps+this_tick-next_tick) ;
   
   next_tick = SDL_GetTicks() + (1000/fps) ; 
   return( realfps) ; /* return the real fps */
}

int LoopDisplay( void* display_manager)
{
   ((DisplayManager *)display_manager)->Loop();
   return 0;
}

