#include <stdlib.h>
#include <stdio.h>
#include <SDL.h>
#include <SDL_thread.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "display.hpp"
#include "midi_lyric.hpp"
#include "gl_particles.hpp"
#include <FTGLExtrdFont.h>

#define FONT_FILE       "arial.ttf"
#define NB_PARTICLES            100
  
class Display_Gl : public Display
{  
 public:
   Display_Gl();
   virtual ~Display_Gl();
   virtual void Initialize();
   virtual void Terminate(); 
   virtual void Refresh( unsigned int dt);
   
 private:
   void DisplayText( int x, int y, Midi_Lyric_Line *line, bool active);
   int current_line;
   int line_state;
   int begining_of_line;
   Gl_Particles *particles;
   FTFont* font;
   void SetCamera();
   void Lighting();
};

Display_Gl::Display_Gl()
{
   state = UNINITIALIZED;
   current_line = 0;
   line_state = 0;
}

Display_Gl::~Display_Gl()
{
   if ( state == WORKING)
     Terminate();
}

void Display_Gl::Initialize()
{  
   char *fontfile = FONT_FILE;
   
   // 1) Load fonts
   
   font = new FTGLExtrdFont( fontfile);
    
   /* TODO  if( !fonts->Open( fontfile, false))
     {
	fprintf( stderr, "Display_Gl::Initialize: failed to open font %s\n", fontfile);
	return;
     }*/
   
   if( !font->FaceSize( 32))
     {
	fprintf( stderr, "Display_Gl::Initialize: failed to set size\n");
	return;
     }
   
   font->Depth(2);
   
   // 2) Init Particles
   
   particles = new Gl_Particles( NB_PARTICLES, "particle.bmp");
   
   state = WORKING;

   fprintf( stderr, "Display_Gl plugin ready.\n");
}

void Display_Gl::Terminate()
{
   state = UNINITIALIZED;
}

void Display_Gl::Refresh( unsigned int dt_msec)
{
   if ( state != WORKING)
     return;
   
   if( !lyrics_ready )
     return;
   
   // find the new current_line
   
   if ( current_tick < this->current_tick) // we have moved backward
     {
	current_line = 0;
	line_state=0;
	fprintf( stderr, "We have moved forward?\n");
     }
   
   this->current_tick = current_tick;

   begining_of_line = 0;
   while( current_line<lyrics->GetNbLines()-1 && current_tick >= lyrics->GetLine(current_line+1)->GetStartingTime() )
     { 
	current_line++;
	line_state = ( line_state + 1 ) % 4;
	begining_of_line = 1;
     }

   // Draw everything
	   
   SetCamera();
   Lighting();
   glScalef( 2,2,2);
   switch( line_state)
     {
      case 0: // premiere ligne active 
	DisplayText( 0, -1, lyrics->GetLine(current_line), true);
	DisplayText( 0, 0, lyrics->GetLine(current_line+1), false);
	DisplayText( 0, 1, lyrics->GetLine(current_line+2), false);
	DisplayText( 0, 2, lyrics->GetLine(current_line+3), false);
	break;
      case 1: // seconde ligne active
	DisplayText( 0, -1, lyrics->GetLine(current_line-1), false);
	DisplayText( 0, 0, lyrics->GetLine(current_line), true);
	DisplayText( 0, 1, lyrics->GetLine(current_line+1), false);
	DisplayText( 0, 2, lyrics->GetLine(current_line+2), false);
	break;
      case 2: // troisieme ligne active
	DisplayText( 0, -1, lyrics->GetLine(current_line+2), false);
	DisplayText( 0, 0, lyrics->GetLine(current_line+3), false);
	DisplayText( 0, 1, lyrics->GetLine(current_line), true);
	DisplayText( 0, 2, lyrics->GetLine(current_line+1), false);
	break;
      case 3: // quatrieme ligne active
	DisplayText( 0, -1, lyrics->GetLine(current_line+1), false);
	DisplayText( 0, 0, lyrics->GetLine(current_line+2), false);
	DisplayText( 0, 1, lyrics->GetLine(current_line-1), false);
	DisplayText( 0, 2, lyrics->GetLine(current_line), true);
	break;
     }

   particles->Update( dt_msec); 
   
   glShadeModel( GL_SMOOTH);
   glDisable( GL_DEPTH_TEST);
   glEnable( GL_BLEND);
   glBlendFunc( GL_SRC_ALPHA, GL_ONE);
   glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
   glHint( GL_POINT_SMOOTH_HINT, GL_NICEST);

   particles->Draw();
}

void Display_Gl::DisplayText( int x, int y, Midi_Lyric_Line *line, bool active)
{
   float full_advance = 0.0f;
   char  colored_string[1024];
   float colored_string_advance = 0.0f;
   float scale_factor = 1.0f;
   
   if ( line == NULL)
     return;
   
   glPushMatrix();
   
   full_advance = font->Advance( line->characters);
   
   if ( full_advance > 400.)//screen->w )
     {
	scale_factor = 400. / (double)full_advance;
	glScalef( scale_factor, 1.0f, 1.0f);
     }
   
   // display colored part

   int separation = line->GetSeparation( current_tick);
   
   if ( separation > 0)
     {
	glPushMatrix();
	strncpy( colored_string, line->characters, separation);
	colored_string[separation] = '\0';
	colored_string_advance = font->Advance( colored_string);
	glDisable( GL_LIGHT0);
	glTranslatef( -full_advance/2, -y*40, 0.);
	glColor4f( 1., 0.2, 0.2, 1.);
	font->Render( colored_string);
	glEnable( GL_LIGHT0);
	glPopMatrix();
     }
   
   // display uncolored part
   
   if ( separation < line->nb_characters)
     {
	glPushMatrix();
	glTranslatef(-full_advance/2+colored_string_advance, -y*40, 0.);
	glColor4f( 1.,1.,1.,1.);
	font->Render( line->characters+separation);
	glPopMatrix();
     }

   if ( active )
     {
	if ( separation == 0)
	  particles->SetPosition( (-full_advance/2+colored_string_advance)*scale_factor, -y*40, 0.0f);
	else if ( separation < line->nb_characters)
	  particles->MoveTo( (-full_advance/2+colored_string_advance)*scale_factor, -y*40, 0.0f, 0.1f);
	else
	  particles->MoveTo( (-full_advance/2+full_advance)*scale_factor, -y*40, 0.0f, 0.1f);
     }
   
   glPopMatrix();
}

void Display_Gl::SetCamera()
{
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   glClearColor( 0., 0., 0., 0.);
   glColor3f( 1.0, 1.0, 1.0);
	
   glEnable( GL_CULL_FACE);
   glFrontFace( GL_CCW);
   
   glEnable( GL_DEPTH_TEST);
   
   //   glEnable( GL_POLYGON_OFFSET_LINE);
   //   glPolygonOffset( 1.0, 1.0); // ???

   ////////////////
   
   glMatrixMode (GL_PROJECTION);
   glLoadIdentity ();
   gluPerspective( 90, 640 / 480, 1, 100000);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   gluLookAt( 0.0, 0.0, 480, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);

}

void Display_Gl::Lighting()
{
   // Set up lighting.
   float light1_ambient[4]  = { 1.0, 1.0, 1.0, 1.0 };
   float light1_diffuse[4]  = { 1.0, 0.9, 0.9, 1.0 };
   float light1_specular[4] = { 1.0, 0.7, 0.7, 1.0 };
   float light1_position[4] = { -1.0, 1.0, 1.0, 0.0 };
   glLightfv(GL_LIGHT1, GL_AMBIENT,  light1_ambient);
   glLightfv(GL_LIGHT1, GL_DIFFUSE,  light1_diffuse);
   glLightfv(GL_LIGHT1, GL_SPECULAR, light1_specular);
   glLightfv(GL_LIGHT1, GL_POSITION, light1_position);
   glEnable(GL_LIGHT1);

   float light2_ambient[4]  = { 0.2, 0.2, 0.2, 1.0 };
   float light2_diffuse[4]  = { 0.9, 0.9, 0.9, 1.0 };
   float light2_specular[4] = { 0.7, 0.7, 0.7, 1.0 };
   float light2_position[4] = { 1.0, -1.0, -1.0, 0.0 };
   glLightfv(GL_LIGHT2, GL_AMBIENT,  light2_ambient);
   glLightfv(GL_LIGHT2, GL_DIFFUSE,  light2_diffuse);
   glLightfv(GL_LIGHT2, GL_SPECULAR, light2_specular);
   glLightfv(GL_LIGHT2, GL_POSITION, light2_position);
   glEnable(GL_LIGHT2);

   float front_emission[4] = { 0.3, 0.2, 0.1, 0.0 };
   float front_ambient[4]  = { 0.2, 0.2, 0.2, 0.0 };
   float front_diffuse[4]  = { 0.95, 0.95, 0.8, 0.0 };
   float front_specular[4] = { 0.6, 0.6, 0.6, 0.0 };
   glMaterialfv(GL_FRONT, GL_EMISSION, front_emission);
   glMaterialfv(GL_FRONT, GL_AMBIENT, front_ambient);
   glMaterialfv(GL_FRONT, GL_DIFFUSE, front_diffuse);
   glMaterialfv(GL_FRONT, GL_SPECULAR, front_specular);
   glMaterialf(GL_FRONT, GL_SHININESS, 16.0);
   glColor4fv(front_diffuse);

   glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
   glEnable(GL_CULL_FACE);
   glColorMaterial(GL_FRONT, GL_DIFFUSE);
   glEnable(GL_COLOR_MATERIAL);

   glEnable(GL_LIGHTING);
   glShadeModel(GL_SMOOTH);
 }

Display_Gl DYNAMIC_DISPLAY_SYMBOL;
