#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/select.h>
#include <errno.h>
#include <time.h>

#include "libdg100.h"

#define RESPONSE_BUFFER_SIZE (2+2+1024+2+2+1)

struct dg100 
{
   char *device_filepath;
   int device_fd;
   struct termios *initial_termios_p; 
   
   unsigned char response[RESPONSE_BUFFER_SIZE];
   unsigned short response_length;

   dg100_track * tracks;
   unsigned int tracks_size;
   unsigned int tracks_length;
   unsigned int tracks_cursor;
   
   int track_style;
   unsigned int cursor;
   
   dg100_waypoint * waypoints;
   unsigned int waypoints_size;
   unsigned int waypoints_length;
   unsigned int waypoints_cursor;
};

unsigned char dg100_msg_get_file_info[] = { 
   0xBB, 0x00, 0x00 
};

unsigned char dg100_msg_enter_gmouse_mode[] = { 
   0xBC, 0x01
};

void dg100_send_cmd( dg100 self, unsigned char *payload, unsigned short payload_length);

void dg100_read_response( dg100 self, unsigned short length);

void dg100_parse_track( dg100 self, unsigned char *data, dg100_waypoint * waypoint);

dg100 dg100_create( char *device)
{
   struct dg100 * self = NULL;
   struct termios termios_p;
   
   self = ( struct dg100 * ) malloc ( sizeof (struct dg100) );

   self->device_filepath = device;

   self->device_fd = open ( self->device_filepath, O_RDWR);
   
   if ( self->device_fd == -1 )
     {
	fprintf ( stderr, "dg100_connnect: %s\n", strerror( errno));
	exit(EXIT_FAILURE);
     }
   
   tcgetattr( self->device_fd, &termios_p);

   cfmakeraw( &termios_p);

   cfsetispeed ( &termios_p, B115200);
   
   cfsetospeed ( &termios_p, B115200);
      
   tcsetattr ( self->device_fd, TCSAFLUSH, &termios_p);

   self->response_length = 0;

   self->tracks = (dg100_track *) malloc(sizeof(struct dg100_track)*1024);
   self->tracks_size = 1024;
   self->tracks_length = 0;
      
   self->waypoints = (dg100_waypoint *) malloc(sizeof(struct dg100_waypoint)*1024);
   self->waypoints_size = 1024;
   self->waypoints_length = 0;
   
   return self;
}


void dg100_get_status( dg100 self)
{
}

void dg100_get_id( dg100 self)
{
}

void dg100_set_id( dg100 self)
{
}

void dg100_get_config( dg100 self)
{
}

void dg100_set_config( dg100 self)
{
}

void dg100_enter_gmouse_mode( dg100 self)
{
   dg100_send_cmd( self, dg100_msg_enter_gmouse_mode, sizeof(dg100_msg_enter_gmouse_mode));
}

void dg100_leave_gmouse_mode( dg100 self)
{
   
}

void dg100_get_tracks_list( dg100 self)
{
   unsigned char buf[3];
   unsigned short idx = 0;
   unsigned short cntInfoCur = 0;
   
   self->tracks_length = 0;
   self->tracks_cursor = 0;
   
   buf[0] = 0xbb;
   buf[1] = 0;
   buf[2] = 0;
   
   dg100_send_cmd( self, buf, 3);
   
   dg100_read_response( self, 2+2+1024+2+2);
      
   if ( self->response_length > 0 )
     {
	int response_type = self->response[4];
	
#ifdef _DEBUG
	
	fprintf ( stderr, "got %d bytes\n", self->response_length);
	
#endif
	
	if ( response_type == 0xbb )
	  {
	     int i;
	     	     
	     idx = self->response[5] * 256 + self->response[6];
	     
	     cntInfoCur = self->response[7] * 256 + self->response[8];
	
//	     fprintf ( stderr, "File Headers received : %d / %d\n", idx, cntInfoCur);
	     
	     for ( i = 0 ; i < cntInfoCur ; i++)
	       {
		  unsigned int i1, i2, i3;
		  		  
		  i1 = (((unsigned int) self->response[ 9+i*12]) * 16777216 ) | (((unsigned int) self->response[10+i*12]) * 65536 ) | (((unsigned int) self->response[11+i*12]) * 256 ) | (((unsigned int) self->response[12+i*12]) );
		  i2 = (((unsigned int) self->response[13+i*12]) * 16777216 ) | (((unsigned int) self->response[14+i*12]) * 65536 ) | (((unsigned int) self->response[15+i*12]) * 256 ) | (((unsigned int) self->response[16+i*12]) );
		  i3 = (((unsigned int) self->response[17+i*12]) * 16777216 ) | (((unsigned int) self->response[18+i*12]) * 65536 ) | (((unsigned int) self->response[19+i*12]) * 256 ) | (((unsigned int) self->response[20+i*12] ));
		  		  
		    {
		       
		       int hh = i1 / 10000;
		       int mm = (i1 - hh * 10000) / 100;
		       int ss = i1 - hh * 10000 - mm * 100;
		       int DD = i2 / 10000;
		       int MM = (i2 - DD * 10000) / 100;
		       int YY = 2000 + i2 - DD * 10000 - MM * 100;
			       
		       self->tracks[i].datetime.tm_sec = ss;
		       self->tracks[i].datetime.tm_min = mm;
		       self->tracks[i].datetime.tm_hour = hh;
		       self->tracks[i].datetime.tm_mday = DD;
		       self->tracks[i].datetime.tm_mon = MM - 1;
		       self->tracks[i].datetime.tm_year = YY - 1900;
		       self->tracks[i].datetime.tm_wday = 0;
		       self->tracks[i].datetime.tm_yday = 0;
		       self->tracks[i].datetime.tm_isdst = 0;
		       
		       self->tracks[i].index = i;
		               
//		       fprintf ( stderr, "[%d] %04d-%02d-%02d um %02d:%02d (UTC) ab %d \n", i, YY, MM, DD, hh, mm, i3);
	
		       self->tracks_length += 1;
		    }
		  
	       }
	  }
	else
	  {
	     fprintf ( stderr, "dg100_get_tracks_list: unknow response type: %02x\n", response_type);
	  }
	
     }
   
   
}

void dg100_rewind_track( dg100 self)
{
   self->tracks_cursor = 0;
}

dg100_track * dg100_get_next_track( dg100 self)
{
   if ( self->tracks_cursor < self->tracks_length )
     {
	return self->tracks+(self->tracks_cursor++);
     }
   else
     {
	return NULL;
     }  
}



void dg100_get_track( dg100 self, unsigned short index)
{
   unsigned char buf[3];
   int i = 0;
   unsigned char track_data[32];
   int track_length = 0;
   int end_reached = 0;
   
   self->waypoints_length = 0;
   self->waypoints_cursor = 0;

   buf[0] = 0xb5;
   buf[1] = index / 256;
   buf[2] = index % 256;
   
   dg100_send_cmd( self, buf, 3);
   
   dg100_read_response( self, 2+2+1024+2+2);

//   fprintf(stderr,"got response 1 (%d bytes)\n", self->response_length);
   
   if ( self->response[4] == 0xb5)
     {
//	fprintf( stderr, "Got a track file\n");  
	
	// get the first track (c styled)

	self->track_style = 2;
	
	dg100_parse_track( self, self->response+5, self->waypoints);
	
	if ( self->track_style == 0 )
	  {
	     track_length = 8;
	  }
	else if ( self->track_style == 1 )
	  {
	     track_length = 20;
	  }
	else if ( self->track_style == 1 )
	  {
	     track_length = 32;
	  }
	else	  
	  {
	     fprintf ( stderr, "dg100_get_track: unkonw track style\n");
	     return;
	  }
	
	self->cursor = 5+32;
	
	self->waypoints_length = 1;

	while ( ! end_reached)
	  {
	     if ( self->cursor + track_length >= self->response_length-3 )
	       {
		  int first_part_length = self->cursor + track_length - (self->response_length-4);
		  int second_part_length = track_length - first_part_length;
		  
//		  fprintf(stderr,"end response (%d bytes / first_part_length=%d)\n", self->cursor, first_part_length);
		  
		  memcpy( track_data, self->response+self->cursor, first_part_length);
		  
		  dg100_read_response( self, 2+2+1024+2+2);
		  
		  if ( self->response_length == 0 )
		    {
//		       fprintf(stderr,"end track\n", self->cursor);
		       return;
		    }
		  
		  memcpy( track_data+first_part_length, 5+self->response, second_part_length);
		  
		  self->cursor = 5+second_part_length;
		 
	       }
	     else
	       {
	       
		  memcpy ( track_data, self->response+self->cursor, track_length);
		  self->cursor += track_length;
	       }
	     
	     if ( track_data[0] == 0xff && track_data[1] == 0xff && track_data[2] == 0xff && track_data[3] == 0xff && track_data[4] == 0xff && track_data[5] == 0xff )
	       {
//		  fprintf ( stderr, "track completed.\n");
		  end_reached = 1;
	       }
	     else
	       {
		  
		  dg100_parse_track( self, track_data, self->waypoints+self->waypoints_length);
		  
		  self->waypoints_length += 1;  
	       }
	  }
	
	
     
     }

}

void dg100_rewind_waypoint( dg100 self)
{
   self->waypoints_cursor = 0;
}

dg100_waypoint * dg100_get_next_waypoint( dg100 self)
{
   if ( self->waypoints_cursor < self->waypoints_length )
     {
	return self->waypoints+(self->waypoints_cursor++);
     }
   else
     {
	return NULL;
     }  
}

void dg100_get_delete_tracks( dg100 self)
{
   
}

void dg100_send_cmd( dg100 self, unsigned char *payload, unsigned short payload_length)
{
   unsigned char buf[2+2+1023+2+2];
   int i = 0;
   unsigned short checksum = 0;
   
   buf[0] = 0xa0;
   buf[1] = 0xa2;   
      
   buf[2] = (unsigned char) (( payload_length & 0xff00 ) >> 8);
   buf[3] = (unsigned char) (( payload_length & 0x00ff ));
   
   for ( i = 0 ; i < payload_length ; i++ )
     {
	buf[4+i] = payload[i];
	checksum += payload[i];
     }
   
   buf[4+i+0] = (unsigned char) (( checksum & 0xff00 ) >> 8);
   buf[4+i+1] = (unsigned char) (( checksum & 0x00ff ));
   buf[4+i+2] = 0xb0;
   buf[4+i+3] = 0xb3;

#ifdef _DEBUG
   
   fprintf(stderr,"send %d bytes: ", 2+2+payload_length+2+2);
   
   for ( i = 0 ; i < 2+2+payload_length+2+2 ; i++ )
     {
	fprintf(stderr,"%02x ", buf[i]);
     }
   
   fprintf(stderr,"\n");
   
#endif
   
   write ( self->device_fd, buf, 2+2+payload_length+2+2);
   
/* tcdrain( self->device_fd); */
   
}

void dg100_read_response( dg100 self, unsigned short length)
{
   fd_set rfds;
   struct timeval tv;
   int retval;
   int no_more_data = 0;
   
   self->response_length = 0;

   while ( self->response_length <= length && no_more_data == 0 )
     {  
	
	FD_ZERO(&rfds);
	FD_SET(self->device_fd, &rfds);
	
	tv.tv_sec  = 2;
	tv.tv_usec = 0;
	
	retval = select( self->device_fd+1, &rfds, NULL, NULL, &tv);
	
	if ( retval == -1 )
	  {
	     perror( "dg100_read_response: select()");
	  }
	else if ( retval )
	  {
	     int n;
	     
	     n = read ( self->device_fd, self->response+self->response_length, RESPONSE_BUFFER_SIZE-self->response_length);
	     
#ifdef _DEBUG 

	       {
		  int i;
		  
		  fprintf(stderr,"readed %d bytes ", n);
		  
		  for ( i = 0 ; i < n ; i++ )
		    {
		       fprintf ( stderr, "%02x ", self->response[self->response_length+i]);
		    }
		  
		  fprintf ( stderr, "\n");
	       }
	     
#endif	  
   
	     self->response_length += n;
	  }
	else
	  {
	     no_more_data = 1;
	  }
     }
   
}



void dg100_parse_track( dg100 self, unsigned char *data, dg100_waypoint * waypoint)
{
   unsigned char *c = data;
   char wg_latitude[256];
   char wg_longitude[256];
  
   int latitude  = ((int)c[0]) * 256 * 256 * 256 + ((int)c[1]) * 256 * 256 + ((int)c[2]) * 256 + ((int)c[3]);
   int longitude = ((int)c[4]) * 256 * 256 * 256 + ((int)c[5]) * 256 * 256 + ((int)c[6]) * 256 + ((int)c[7]);

   int latitude_deg = latitude  / 1000000;
   int longitude_deg = longitude  / 1000000;

   float latitude_min =  ((float)(latitude  - latitude_deg  * 1000000 ) / 10000.0);
   float longitude_min = ((float)(longitude - longitude_deg * 1000000 ) / 10000.0);

//   waypoint->latitude_deg  = ((double)latitude ) / 1000000.;
//   waypoint->longitude_deg = ((double)longitude) / 1000000.;
   
   waypoint->latitude_deg  = ((float)latitude_deg)  + latitude_min  / 60.0;
   waypoint->longitude_deg = ((float)longitude_deg) + longitude_min / 60.0;
     
//   sprintf( wg_latitude,  "%d %c %02.4f (%f) ", latitude/1000000, latitude > 0 ? 'N' : 'S', ((float)(latitude%1000000))/10000.0, latitude_min );
//   sprintf( wg_longitude, "%d %c %02.4f (%f)", longitude/1000000, longitude > 0 ? 'E' : 'W', ((float)(longitude%1000000))/10000.0, longitude_min );
   
//   fprintf ( stderr, "%d %d -> %s %s", latitude, longitude, wg_latitude, wg_longitude);

   if ( self->track_style == 2 || self->track_style == 1 )
     {
	int utime     = ((int)c[8]) * 256 * 256 * 256 + ((int)c[9]) * 256 * 256 + ((int)c[10]) * 256 + ((int)c[11]);
	int udate     = ((int)c[12]) * 256 * 256 * 256 + ((int)c[13]) * 256 * 256 + ((int)c[14]) * 256 + ((int)c[15]);
	int speed     = ((int)c[16]) * 256 * 256 * 256 + ((int)c[17]) * 256 * 256 + ((int)c[18]) * 256 + ((int)c[19]);
	
        int hh = utime / 10000;
	int mm = (utime - hh * 10000) / 100;
	int ss = utime - hh * 10000 - mm * 100;
	int DD = udate / 10000;
	int MM = (udate - DD * 10000) / 100;
	int YY = 2000 + udate - DD * 10000 - MM * 100;
 
	waypoint->datetime.tm_sec = ss;
	waypoint->datetime.tm_min = mm;
	waypoint->datetime.tm_hour = hh;
	waypoint->datetime.tm_mday = DD;
	waypoint->datetime.tm_mon = MM - 1;
	waypoint->datetime.tm_year = YY - 1900;
	waypoint->datetime.tm_wday = 0;
	waypoint->datetime.tm_yday = 0;
	waypoint->datetime.tm_isdst = 0;
	
//	fprintf ( stderr, " | %04d-%02d-%02d %02d:%02d:%02d", YY, MM, DD, hh, mm, ss);
     }
   
   if ( self->track_style == 2 )
     {
	int altitude  = c[20] * 256 * 256 * 256 + c[21] * 256 * 256 + c[22] * 256 + c[23];
	int style     = c[28] * 256 * 256 * 256 + c[29] * 256 * 256 + c[30] * 256 + c[31];
	
//	fprintf ( stderr, " | %d %d", altitude, style);
	
	self->track_style = style;	
     }
   
//   fprintf ( stderr, "\n");
}
