#include <stdlib.h>
#include <stdio.h>
#include "libdg100.h"

#define DEFAULT_DEVICE         "/dev/ttyUSB0"
#define DEFAULT_EXPORT_FORMAT  "gpx"

typedef enum 
{
   LIST_TRACKS = 0 ,
   EXPORT_SINGLE_TRACK = 1,
   EXPORT_ALL_TRACKS = 2,
   DELETE_ALL_TRACKS = 3  
} action_t;

typedef enum Action actions;


void export_track_as_gpx( dg100 gps_logger, int track_index)
{
   FILE *f_out = NULL;
   float max_lat = -90., min_lon = 180., min_lat = 90., max_lon = -180.;
   dg100_waypoint *waypoint = NULL;
   char output_filename[256];
   char datetime[128];
   
   dg100_get_track( gps_logger, track_index);

   dg100_rewind_waypoint( gps_logger);
   
   waypoint = dg100_get_next_waypoint( gps_logger);
   
   if ( waypoint == NULL )
     {
	fprintf ( stderr, "track %d: Empty track skeeped\n", track_index);
	return;
     }
   
   strftime( datetime, 127, "%Y-%m-%d %T", &(waypoint->datetime));
   
   sprintf ( output_filename, "track-%s.gpx", datetime);
   
   fprintf ( stdout, "exporting track %d to %s\n", track_index, output_filename);
   
   do 
     {
	if ( waypoint->latitude_deg > max_lat )
	  max_lat = waypoint->latitude_deg;
	if ( waypoint->latitude_deg < min_lat )
	  min_lat = waypoint->latitude_deg;
	if ( waypoint->longitude_deg > max_lon )
	  max_lon = waypoint->longitude_deg;
	if ( waypoint->longitude_deg < min_lon )
	  min_lon = waypoint->longitude_deg;
     }
   while ( waypoint = dg100_get_next_waypoint( gps_logger) );
   
   dg100_rewind_waypoint( gps_logger);
   
   f_out = fopen (output_filename,"w");
   
   if ( f_out == NULL )
     {
	return;
     }
   
   fprintf ( f_out, "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\" ?>\n");
   fprintf ( f_out, "<gpx xmlns=\"http://www.topografix.com/GPX/1/1\" creator=\"dg100\" version=\"1.1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd\">\n");
   fprintf ( f_out, "\n");
   fprintf ( f_out, " <metadata>\n");
   fprintf ( f_out, "  <link href=\"http://duberga.net/dg100\">\n");
   fprintf ( f_out, "   <text>dg100 utils</text>\n");
   fprintf ( f_out, "  </link>\n");
   fprintf ( f_out, "  <time>2007-09-20T12:00:00Z</time>\n");
   fprintf ( f_out, "  <bounds minlat=\"%f\" minlon=\"%f\" maxlat=\"%f\" maxlon=\"%f\"/>\n", min_lat, min_lon, max_lat, max_lon);
   fprintf ( f_out, " </metadata>\n");
   fprintf ( f_out, "\n");
   fprintf ( f_out, " <trk>\n");
   fprintf ( f_out, "  <name>LOG</name>\n");
   fprintf ( f_out, "  <trkseg>\n");
   
   while ( waypoint = dg100_get_next_waypoint( gps_logger) )
     {
	fprintf ( f_out, "   <trkpt lat=\"%f\" lon=\"%f\">\n", waypoint->latitude_deg, waypoint->longitude_deg);
	fprintf ( f_out, "    <ele>0.0</ele>\n");
	fprintf ( f_out, "    <time>2007-09-2ßT12:00:00Z</time>\n");
	fprintf ( f_out, "   </trkpt>\n");
     }
   
   fprintf ( f_out, "  </trkseg>\n");
   fprintf ( f_out, " </trk>\n");
   fprintf ( f_out, "\n");
   fprintf ( f_out, "</gpx>\n");
		     
   fclose (f_out);
}

int main ( int argc, char **argv)
{
   char *device_filepath = DEFAULT_DEVICE;
   action_t action = LIST_TRACKS;
   int track_index = 0;
   dg100 gps_logger = NULL;
   dg100_track *track = NULL;
   int a = 0;
   char datetime[128];
   
   for( a = 1 ; a < argc ; a++) 
     {
	if ( strcmp( argv[a], "--help") == 0 ) 
	  {
	     fprintf( stderr, "usage: %s [options] <command> \n\n", argv[0]) ;
	     fprintf( stderr, "options are:\n") ;
	     fprintf( stderr, "  --help                    display this help notice\n") ;
	     fprintf( stderr, "  --device <device>         DG100 serial device (default %s)\n", DEFAULT_DEVICE);
	     fprintf( stderr, "  --export-format <format>  use an alternate output fprmat (default %s)\n", DEFAULT_EXPORT_FORMAT);
	     fprintf( stderr, "\n");
	     fprintf( stderr, "commands are:\n") ;
	     fprintf( stderr, "  list-tracks               list all available tracks\n") ;
	     fprintf( stderr, "  export-track <index>      export a single track\n");
	     fprintf( stderr, "  export-all-tracks         export a single track\n");
	     fprintf( stderr, "  delete-all-tracks         free DG100 memory\n");
	     	     
	     exit( EXIT_SUCCESS) ;
	  }
	else if ( strcmp( argv[a], "--device") == 0 )
	  {
	     if ( a+1 < argc )
	       {
		  device_filepath = argv[++a];
	       }
	     else
	       {
		  fprintf( stderr, "--device parameter need one argument, please consult help with --help\n");
		  return EXIT_FAILURE;
	       }    
	  }
	else if ( strncmp( argv[a], "-", 1) == 0 )
	  {
	     fprintf( stderr, "%s: unknow parameter %s\n", argv[0], argv[a]);
	  }
	
	// that's a command
	     
	else if ( strcmp( argv[a], "list-tracks") == 0 )
	  {
	     action = LIST_TRACKS;	     
	  }
	else if ( strcmp( argv[a], "export-track") == 0 )
	  {
	     if ( argc <= a+1 )
	       {
		  fprintf( stderr, "I don't know which track you want to export. please consult help with --help\n");
		  exit( EXIT_FAILURE);
	       }

	     track_index = atoi( argv[++a]);	     
	     
	     action = EXPORT_SINGLE_TRACK;
	     
	  }
	else if ( strcmp( argv[a], "export-all-tracks") == 0 )
	  {
	     action = EXPORT_ALL_TRACKS;
	  }
	else if ( strcmp( argv[a], "delete-all-tracks") == 0 )
	  {
	     action = DELETE_ALL_TRACKS;
	  }
	
     }
   
   
   
   gps_logger = dg100_create( device_filepath);
   
   switch ( action )
     {
      case LIST_TRACKS:
	
	dg100_get_tracks_list( gps_logger);
	
	while ( track = dg100_get_next_track(gps_logger) )
	  {
	     strftime( datetime, 127, "%Y-%m-%d %T", &(track->datetime));
	     
	     fprintf ( stdout, "[%d] %s (UTC)\n", track->index, datetime);
	  }
	
	break;
	
      case EXPORT_SINGLE_TRACK:
	
	export_track_as_gpx( gps_logger, track_index);
	  
	break;
      
      case EXPORT_ALL_TRACKS:

	dg100_get_tracks_list( gps_logger);
	
	while ( track = dg100_get_next_track( gps_logger) )
	  {
	     export_track_as_gpx( gps_logger, track->index);
	  }

	
	break;	
     }
   
   exit (EXIT_SUCCESS);
}
